/*****************************************************************************************************
RVC2011 RT-Component
Copyright (c) 2011, Kyoto Univ.
All rights reserved.

Contact us if you use this software for sell.
If you use this software not for sell, you can use this software under BSD lisence.
See the files LICENSE.TXT and LICENSE-BSD.TXT for more details.                     
*****************************************************************************************************/
/*****************************************************************************************************/
/*!
*	@file GlobalDWA.cpp
*	@brief Class for Trajectory generation
*	@author goto@uec, kon@kyoto-u
/*****************************************************************************************************/

#ifndef __GlobalDWA_H
#define __GlobalDWA_H


#define _USE_MATH_DEFINES
#include <math.h>
#include <iostream>
#include "myBitmap.h"
#include "myTimer.h"
#include <vector>
#include <algorithm>
#include <utility> 
#include <functional>
#include "CPose2D.h"
#include "CVelocity2D.h"

using namespace std;
using namespace ptrc::poses;

#define GLOBALSIZE_W  310		//!< GLOBALSIZE_W * GRIDSIZE = About 40 [m]
#define GLOBALSIZE_H  310		//!< GLOBALSIZE_W * GRIDSIZE = About 40 [m]
#define LOCALSIZE_W	   81		//!< [Count Number]  => LOCALSIZE_W * GRIDSIZE = About 40 [m]
#define LOCALSIZE_H	   81		//!< [Count Number]  => LOCALSIZE_W * GRIDSIZE = About 40 [m]

static int OBS_NF       = 999;  //!< Value of obstacle
static int INI_NF       = -1;	//!< Initial value of NF map
static double GRIDSIZE	= 0.5;	//!< Grid size 


/*****************************************************************************************************/
/*!
*	@struct aroundNF
*	@brief to store the NF valus in surrounding cell.
*/
/*****************************************************************************************************/
struct aroundNF
{
	double center;
	double right;
	double left;
	double upper;
	double lower;
};

/*****************************************************************************************************/
/*!
*	@struct pixelPos
*	@brief position of pixel
*/
/*****************************************************************************************************/
struct pixelPos
{
	int Width;
	int Height;
};

/*****************************************************************************************************/
/*!
*	@struct mapPixelSize
*	@brief 
*/
/*****************************************************************************************************/
struct mapPixelSize
{
	double W;
	double H;
};

/*****************************************************************************************************/
/*!
*	@struct trajectory
*	@brief store the 
*/
/*****************************************************************************************************/
struct trajectory
{
	double obj;
	CPose2D poseGlobal;
	CPose2D poseLocal;
	CVelocity2D v;
	CVelocity2D v0;
};

/*****************************************************************************************************/
/*!
*	@brief cellInformation
*/
/*****************************************************************************************************/
class cellInformation
{
public:
	/*!
	*	@brief Constructor
	*/
	cellInformation(){clear();}

	/*!
	*	@brief Destructor
	*/
	~cellInformation(){}
	
	/*!
	*	@brief Clear all cell with INI_NF
	*/
	void clear(void)
	{
		NF.center = INI_NF;
		NF.right  = INI_NF;
		NF.left   = INI_NF;
		NF.upper  = INI_NF;
		NF.lower  = INI_NF;
		grad = 0;
		const_vel = 0;
	}

public:
	aroundNF NF;	  //!< NF values of the cells around this cell
	double grad;	  //!< Gradient of NF
	double const_vel; //!< Admissible velocity[m/s]
};

/*******************************************************************************************/
/*!
*	@class Robot_Infomation
*/
/*******************************************************************************************/
class robotSpecification
{
public:
	double width;	//!<Width of the robot[m]
	double vmax;	//!< Maximum velocity of the robot[m/sec]
	double wmax;	//!< Maximum angular velocity of the robot[rad/sec]
	double avmax;	//!< Maximum acceralation of the robot[m/s^2]
	double awmax;	//!< Maximum angular acceralation of the robot[rad/s^2]
	double tsamp;	//!< Smapling period[sec]

	/*!
	*	@brief Default constructor
	*/
	robotSpecification()
	{
		clear();
	};

	/*!
	*	@brief Destructor
	*/
	~robotSpecification(){};

	/*!
	*	@brief Set specification of the robot
	*	@param[in] _width Width of the robot
	*	@param[in] _vmax  Maximum velocity of the robot[m/sec]
	*	@param[in] _wmax  Maximum rotational velocity of the robot[rad/sec]
	*	@param[in] _avmax Maximum accelaration of the robot[m/sec^2]
	*	@param[in] _awmax Maximum accelaration of the robot[rad/sec^2]
	*	@param[in] _tsamp Sampling period[sec]
	*/
	void setSpecification(double _width,double _vmax,double _wmax, double _avmax, double _awmax,double _tsamp)
	{
		width = _width;
		vmax  = _vmax;
		wmax  = _wmax;
		avmax = _avmax;
		awmax = _awmax;
		tsamp = _tsamp;
	}
	/*!
	*	@brief Set iniital value to the member variables
	*/
	void clear()
	{
		width = 0.5;
		vmax  = 0.5;
		wmax  = 0.5;
		avmax = 0.1;
		awmax = 0.1;
		tsamp = 0.1;
	}
};

/*******************************************************************************************/
/*!
*	@class mapInformation
*/
/*******************************************************************************************/
class mapInformation
{
public:

	pixelPos gridnum;		//!< [pixel]
	mapPixelSize gridsize;	//!< [m]

	pixelPos lgridnum;		//!< [pixel]
	mapPixelSize lgridsize;	//!< [m]

	pixelPos waypoint[256];//Pixel position of waypoint. First one is start point, last one is goal point.

	int waypoint_num_all;	//Number of waypoints
	int waypoint_target;	//ID of waypoint (start from 0)

	/*!
	*	@brief Default constructor
	*/
	mapInformation()
	{
		clear();
	};

	/**
	*	@brief Destructor
	*/
	~mapInformation(){};
	
	/*!
	*	@brief clear all member variables
	*/
	void clear()
	{
		gridnum.Width = 0;
		gridnum.Height= 0;
		gridsize.W    = 0;
		gridsize.H    = 0;
		for(int i=0;i<256;i++)
		{
			waypoint[i].Width = 0;
			waypoint[i].Height = 0;
		}
		waypoint_num_all = 0;
		waypoint_target  = 0;
	}
 
	/*!
	*	@brief Set waypoint information
	*	@param[in] wp_w0 
	*	@param[in] wp_h0 
	*	@param[in] wp_num0 
	*/
	void setWaypoints(int* wp_w0,int* wp_h0,int wp_num0)
	{
		waypoint_num_all = wp_num0;
		for(int i=0;i<waypoint_num_all;i++)
		{
			waypoint[i].Width  = wp_w0[i];
			waypoint[i].Height = wp_h0[i];
		}
	}
};

/*******************************************************************************************/
/*!
*	@class acceralation
*	@brief store acceralation values
*/
/*******************************************************************************************/
class acceralation
{
public:
	acceralation()
	{
		av = 0;
		aw = 0;
	}
	acceralation(double _av,double _aw)
	{
		av = _av;
		aw = _aw;
	}
	double av;
	double aw;
};

typedef std::pair<double,trajectory> trajectory_pair;

/*******************************************************************************************/
/*!
*	@class GlobalDWA
*	@brief Class for trajectory generation based on GDWA
*/
/*******************************************************************************************/
class GlobalDWA
{
public:
	cellInformation cell[GLOBALSIZE_W][GLOBALSIZE_H];						//!< loaded 
	cellInformation local[LOCALSIZE_W][LOCALSIZE_H];	//!< Created local map
	mapInformation bp;				//!< loaded raw map information
	myBitmap bmp;					//!< Bitmap manager
	robotSpecification robotSpec;	//!< Specification of the robot

	double a,b,c,d;			//!< weight of objective function
	double Lmax;			//!< Acc/Deacc region width
	CPoint2D targetxy;		//!< Reference position 

	int    emergencyCount;	//!< (if there is no way, then this becomes larger than 0)
	char   emergencyRotationDirection;	//!< Rotation direction when there is no way. ('L' or 'R')

	double T;				//!< Prediction length[sec]
	int    horizon_num;		//!< Number of prediction horizon(i.e. Prediction time is Tsimu*horizon_num)
	int    n;			    //!< 'n' is the 'n' in the paper
	double margin;			//!< Convergence radious 

	int curMapNum;			//!< Current ID of map
	CPose2D offset;			//!< Offset (offset from GL to MapGL)
	CPose2D g_odo;			//!< Robot position in map global (MapGL)frame
	CPoint2D g_target;		//!< Reference position in map global(MapGL) frame
	CPoint2D pre_g_target;	//!< Previous reference position in global frame(MapGL)
	CVelocity2D velPre;		//!< Previous velocity
	bool flag_new_waypoint;	//!< It becomes true when waypoint is revised
	bool flag_new_image;	//!< It becomes true when the robot arrives the last waypoint of an image(it means "Change the map")

	/*!
	*	@brief Default constructor
	*/
	GlobalDWA()
	{
		flag_new_waypoint = true;
		flag_new_image    = true;
		curMapNum         = 0;
		T                 = 1.0;
		horizon_num       = 10;
		n                 = 20;
		margin            = 2.0;
		Lmax              = 1.0;
		a                 = 0.25;
		b                 = 0.25;
		c                 = 0.25;
		d                 = 0.25;
		initializeParameters();
	}

	/*!
	*	@brief Set margin of the convergence radious
	*	@param[in] _margin 
	*/
	void setMargin(double _margin)
	{
		margin = _margin;
	}

	/*!
	*	@brief Generate velocity command based on GDWA
	*	@retval Planned velocity
	*/
	CVelocity2D calculateGDWA(void);

	/*!
	*	@brief Get NF value in local map
	*	@param[in]  x x coordinate in local map
	*	@param[in]  y y coordinate in local map
	*	@retval NF value
	*/
	double getNFLocal(double x,double y);
	
	/*!
	*	@brief get NF value in global map
	*	@param[in]  x x coordinate in global map
	*	@param[in]  y y coordinate in global map
	*	@retval NF value
	*/
	double getNFGlobal(double x,double y);

	/*!
	*	@brief Get velocity contraint information in local map
	*	@param[in]  x x coordinate in local map
	*	@param[in]  y y coordinate in local map
	*	@retval velocity constraint[m/sec]
	*/
	double getConstVelLocal(double x,double y);

	/*!
	*	@brief Get velocity contraint information in global map
	*	@param[in]  x x coordinate in global map
	*	@param[in]  y y coordinate in global map
	*	@retval velocity constraint[m/sec]
	*/
	double getConstVelGlobal(double x,double y);

	/*!
	*	@brief Get NF valus in the surroundings of th cell
	*	@param[in]  x x coordinate in local map
	*	@param[in]  y y coordinate in local map
	*	@retval NF values in the surrounding cells.
	*/
	aroundNF GetNFall(double x,double y);

	/*!
	*	@brief Get gradient
	*/
	double getGradient(double x,double y);

	/*!
	*	@brief 
	*/
	pixelPos GetCellNum_L(double x,double y);
	
	/*!
	*	@brief 
	*/	
	pixelPos GetCellNum_G(double x,double y);
	
	/*!
	*	@brief 
	*/	
	int getWayPointNum(double x,double y,int cur_num);

	/*!
	*	@brief 
	*/
	CVelocity2D trajectoryPlanning(void);

	/*!
	*	@brief 
	*/
	double dNF1(double x,double y);//p

	/*!
	*	@brief 
	*/
	CPoint2D getXY(pixelPos cell);

	/*!
	*	@brief 
	*/
	CPoint2D getXY_L(pixelPos cell);

	/*!
	*	@brief Convert data to
	*/
	void calculatePositionGL(double _x, double _y, double _theta);
 

	/*!
	*	@brief Set obstacle data to map(polar coordinate)
	*	@param[in] obst0 
	*	@param{in} theta 
	*/
	void setObst(double obst0, double theta);
	
	/*!
	*	@brief Set obstacle data to map
	*	@param[in] _x 
	*	@param[in] _y 
	*/
	void setObstacle(double _x, double _y);

	/*!
	*	@brief Set obstacle data to map
	*	@param[in] _x 
	*	@param[in] _y 
	*/
	pixelPos getID(CPoint2D pos);

	/*!
	*	@brief Load map 
	*	@param[in] name Name of map file 
	*	@param[in] length Length of file name
	*/
	void loadMap(string name,int length);
	
	/*!
	*	@brief 
	*/
	void transLocal(void);

	/*!
	*	@brief Is the position (x,y) on the map
	*	@param[in] x
	*	@param[in] y
	*/
	bool isValid(double x,double y);
	
	/*!
	*	@brief Is the position (x,y) on the map
	*	@param[in] x
	*	@param[in] y
	*/
	double getXFromPixel(int id0);
	
	/*!
	*	@brief Is the position (x,y) on the map
	*	@param[in] x
	*	@param[in] y
	*/
	double getYFromPixel(int id0);

	/*!
	*	@brief Initialize the NF map
	*	@retval 
	*/
	bool initializeNF(void);

	/*!
	*	@brief Initialize the parameters
	*	@retval 
	*/
	bool initializeParameters(void);
	
	/*!
	*	@brief 
	*	@retval 
	*/
	double calculateADRegion(double xl,double yl,double min_const_vel);

	/*!
	*	@brief 
	*	@retval 
	*/
	pixelPos getTargetPosition(void);
	
	
	/*!
	*	@brief 
	*	@retval 
	*/
	bool checkWaypoint(void);

	/*!
	*	@brief Set current velocity of the robot
	*	@param[in] v Translational Velocity
	*	@param[in] w Rotational velocity
	*/
	void setCurrentVelocity(double v, double w)
	{
		velPre.setVW(v,w);
	}

	/*!
	*	@brief Set weights of the value function
	*	@param[in]  _a
	*	@param[in]  _b 
	*	@param[in]  _c
	*	@param[in]  _d
	*/
	void setCostFunctionWeight(double _a, double _b, double _c, double _d)
	{
		a = _a;
		b = _b;
		c = _c;
		d = _d;
	}

	/*!
	*	@brief set parameters related to prediction.
	*	@param[in] _Tsimu Sampling period of one prediction
	*	@param[in] _horizon_num Prediction horizon step
	*	@param[in] _n Number of tragectories to be taken over
	*/
	void setPredictionSpecification(double _T, double _horizon_num, double _n)
	{
		T = _T;
		horizon_num = _horizon_num;
		n = _n;
	}

	/*!
	*	@brief Calculate the size of AD region using parameters
	*/
	void calculateADRegionSize()
	{
		Lmax = 1/(2*robotSpec.avmax)*pow(robotSpec.vmax,2);
	}


private:

	/*!
	*	@brief Check whether curpixel0 is out of course in Local Map.
	*	@param[in] curpixel0 
	*	@retval 1:out of course, 0: within the course 
	*/
	bool isCourseOutLO(pixelPos curpixel0);

	/*!
	*	@brief Check whether curpixel0 is out of course in Local Map.
	*/
	bool isCourseOutGL(pixelPos curpixel0);
	
	/*!
	*	@brief 
	*/
	bool calculateManhattanDistance(void);

	/*!
	*	@brief 
	*/
	bool calculateNFGradient(void);
	/*!
	*	@brief 
	*/
	double PItoPI(double theta0);

	/*!
	*	@brief 
	*/
	double calculateDistance();

	myTimer timer;		//!< 
	double min_dist;	//!< 

	//Convert coordinate
	CPose2D convertFrameGLtoMapGL(CPose2D _pose)
	{
		CPose2D tmp;
		tmp.setX( _pose.getX()*cos(offset.getYaw()) - _pose.getY()*sin(offset.getYaw()) );
		tmp.setY( _pose.getX()*sin(offset.getYaw()) + _pose.getY()*cos(offset.getYaw()) );
		tmp.setYaw( _pose.getYaw() + offset.getYaw());

		while(tmp.getYaw()<=-M_PI) tmp.setYaw( tmp.getYaw() + 2*M_PI );
		while(tmp.getYaw() > M_PI) tmp.setYaw( tmp.getYaw() - 2*M_PI );

		return tmp;
	}


protected:
};


#endif
