/*****************************************************************************************************
RVC2011 RT-Component
Copyright (c) 2011, Kyoto Univ.
All rights reserved.

Contact us if you use this software for sell.
If you use this software not for sell, you can use this software under BSD lisence.
See the files LICENSE.TXT and LICENSE-BSD.TXT for more details.                     
*****************************************************************************************************/
// -*- C++ -*-
/*!
 * @file  RVC2011.cpp
 * @brief TrajectoryPlanning
 * @date $Date$
 *
 * $Id$
 */

#include "RVC2011.h"

// Module specification
// <rtc-template block="module_spec">
static const char* rvc2011_spec[] =
  {
    "implementation_id", "RVC2011",
    "type_name",         "VelConstraint",
    "description",       "TrajectoryPlanning",
    "version",           "0.0.1",
    "vendor",            "Kyoto UNIV.",
    "category",          "Moduel",
    "activity_type",     "PERIODIC",
    "kind",              "DataFlowComponent",
    "max_instance",      "0",
    "language",          "C++",
    "lang_type",         "compile",

    // Configuration variables
	"conf.default.width",         "0.50",
	"conf.default.vmax",          "1.0",
	"conf.default.wmax",          "0.5",
	"conf.default.avmax",         "0.4",
	"conf.default.awmax",         "0.5",
	"conf.default.tsamp",         "0.25",
	"conf.default.a",             "0.45",
	"conf.default.b",             "0.35",
	"conf.default.c",             "0.05",
	"conf.default.d",             "0.15",
	"conf.default.margin",        "2.0",
	"conf.default.flag_OBSTACLE", "0",
	"conf.default.T",             "1.0",
	"conf.default.bunshi_num",    "10",
	"conf.default.n",             "20",
	"conf.default.LoadMapName",   "GridMap/region00.bmp",
	"conf.default.MapNum",        "1",
    ""
  };
// </rtc-template>

/*!
 * @brief constructor
 * @param manager Maneger Object
 */
RVC2011::RVC2011(RTC::Manager* manager)
    // <rtc-template block="initializer">
  : RTC::DataFlowComponentBase(manager),
    m_poseIn("pose", m_pose),
    m_currentVelocityIn("currentVelocity", m_currentVelocity),
    m_obstacleDataIn("obstacleData", m_obstacleData),
    //m_cloudIn("cloud", m_cloud),
	m_velocityOut("velocity", m_velocity)

    // </rtc-template>
{
}

/*!
 * @brief destructor
 */
RVC2011::~RVC2011()
{
}



RTC::ReturnCode_t RVC2011::onInitialize()
{
  // Registration: InPort/OutPort/Service
  // <rtc-template block="registration">
  // Set InPort buffers
  //addInPort("cloud", m_cloudIn);
  addInPort("pose", m_poseIn);
  addInPort("currentVelocity",m_currentVelocityIn);
  addInPort("obstacleData", m_obstacleDataIn);  

  // Set OutPort buffer
  addOutPort("velocity", m_velocityOut);
  
  // Set service provider to Ports
  
  // Set service consumers to Ports
  
  // Set CORBA Service Ports
  
  // </rtc-template>
  // Bind variables and configuration variable
 	bindParameter("width",m_width,"0.5");
 	bindParameter("vmax",m_vmax,"1.0");
 	bindParameter("wmax",m_wmax,"0.5");
 	bindParameter("avmax",m_avmax,"0.4");
 	bindParameter("awmax",m_awmax,"0.5");
 	bindParameter("tsamp",m_tsamp,"0.25");
 	bindParameter("a",m_a,"0.4");  //NF 
 	bindParameter("b",m_b,"0.25"); //GOAL
 	bindParameter("c",m_c,"0.25"); //VEL 
 	bindParameter("d",m_d,"0.1");  //dNF 
 	bindParameter("margin",m_margin,"2.0");
	bindParameter("flag_OBSTACLE",m_flag_OBSTACLE,"0");
 	bindParameter("T",m_T,"0.75");
 	bindParameter("horizon",m_horizon,"20");
 	bindParameter("numberOfCandidate",m_numberOfCandidate,"20");
 	bindParameter("mapName",m_mapName,"GridMap/region00.bmp");
 	bindParameter("numberOfMap",m_numberOfMap,"1");
    
  return RTC::RTC_OK;
}

/*
RTC::ReturnCode_t RVC2011::onFinalize()
{
  return RTC::RTC_OK;
}
*/

/*
RTC::ReturnCode_t RVC2011::onStartup(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/

/*
RTC::ReturnCode_t RVC2011::onShutdown(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/


/*****************************************************************************************/
//onActivate
/*****************************************************************************************/
RTC::ReturnCode_t RVC2011::onActivated(RTC::UniqueId ec_id)
{
	gdwa   = new GlobalDWA;
	isInit = false;

	//Copy parameters to GDWA
	gdwa->robotSpec.setSpecification(m_width,m_vmax,m_wmax,m_avmax,m_awmax,m_tsamp);
	gdwa->setCostFunctionWeight(m_a,m_b,m_c,m_d);
	gdwa->setPredictionSpecification(m_T,m_horizon,m_numberOfCandidate);
	gdwa->calculateADRegionSize();//Calculate the size of AD region
	gdwa->setMargin(m_margin);

	//Initialize flags
	flag_odo 			= false;
	flag_vel            = false;
	flag_obs 			= false;
	flag_time			= false;

	//Set initial time 
	initTime = timer.getTime();
	curTime  = 0;
	preTime  = initTime;

  return RTC::RTC_OK;
}

/*****************************************************************************************/
//onDeactivate
/*****************************************************************************************/
RTC::ReturnCode_t RVC2011::onDeactivated(RTC::UniqueId ec_id)
{
  delete gdwa;
  return RTC::RTC_OK;
}

/*****************************************************************************************/
//onExecute
/*****************************************************************************************/
RTC::ReturnCode_t RVC2011::onExecute(RTC::UniqueId ec_id)
{
	//=====================================================================
	//Brief procedure of trajectory generation
	//1. Get position data of the robot
	//2. Get the obstacle data(LRF data)
	//3. Convert position data into map coordinate
	//4. Set obstacle data to grid map
	//5. Calculate NF and its gradient
	//6. Calculate velocity input based on GDWA
	//7. Output v,w 
	//=====================================================================

	//Receive position data from the robot
	if( m_poseIn.isNew())
	{
		m_poseIn.read();
		if( !isInit )
		{
			initOffset.data.position.x = m_pose.data.position.x;
			initOffset.data.position.y = m_pose.data.position.y;
			initOffset.data.heading = m_pose.data.heading;
			isInit = true;
		}
		else
		{
			flag_odo = true;
			m_pose.data.position.x = m_pose.data.position.x - initOffset.data.position.x;
			m_pose.data.position.y = m_pose.data.position.y - initOffset.data.position.y;
			m_pose.data.heading    = m_pose.data.heading    - initOffset.data.heading;
			while(m_pose.data.heading<=-M_PI) m_pose.data.heading = m_pose.data.heading + 2*M_PI;
			while(m_pose.data.heading > M_PI) m_pose.data.heading = m_pose.data.heading - 2*M_PI;
		}
	}

	//Receive velocity data
	if( m_currentVelocityIn.isNew() )
	{
		m_currentVelocityIn.read();
		flag_vel = true;
	}
		
	//Receive obstacle data
	if( m_obstacleDataIn.isNew() )
	{
		m_obstacleDataIn.read();
		flag_obs = true;
	}
	else if( !m_flag_OBSTACLE )
	{
		flag_obs = true;
	}

	//Get current time
	curTime = timer.getTime();
	if( curTime-preTime >= m_tsamp )
	{
		flag_time = true;
		preTime = curTime;
	}

	//Execute the trajectory generation, if all data are available.
	if( flag_odo && flag_obs && flag_time && flag_vel )
	{
		flag_odo = false;
		flag_obs = false;
		flag_time= false;

		//Convert coordinate to MapGL coordinate(?), and set it to class
		gdwa->calculatePositionGL(m_pose.data.position.x,m_pose.data.position.y,m_pose.data.heading);

		//Check the waypoint, to be followed
		gdwa->checkWaypoint();

		//Current map is terminated, so load ner map
		if(m_numberOfMap > gdwa->curMapNum && gdwa->flag_new_image )
		{
			cout<<"mapName::"<<m_mapName.c_str()<<","<<m_mapName<<",LoadMapName Length::"<<(int)m_mapName.length()<<endl;
			gdwa->loadMap( m_mapName.c_str(), (int)m_mapName.length() );		
		}
		else if(m_numberOfMap<=gdwa->curMapNum && gdwa->flag_new_image)
		{
			//If there are no additional map, then the robot must be at the goal position.
			m_velocity.data.vx = 0;
			m_velocity.data.va = 0;
			m_velocityOut.write();

			cout<<"Arrived at the goal"<<endl;
			return RTC::RTC_OK;
		}

		//initialize local map
		gdwa->initializeNF();

		//Set obstacle data to local map
		if( m_flag_OBSTACLE )
		{
			double laserTheta;  
			double stepLRF   =  ( LaserSensorPrecision * M_PI /180.0 );
			double offsetLRF = -( LaserWide/2 )*M_PI/180.0; 

			for(int i=0;i < (int)(LaserWide/LaserSensorPrecision) ;i++)
			{
				laserTheta = offsetLRF + stepLRF * i; //Angle of each step 
				
				if(  m_obstacleData.data[i]>= 0.2 )
				{
					gdwa->setObst((double)m_obstacleData.data[i]/1000.0, laserTheta); 
				}
			}
		}

		//Convert obstacles, RVCs, waypoints within the local map into local frame.
		gdwa->transLocal();

        //Set current velocity
		gdwa->setCurrentVelocity( m_currentVelocity.data.vx, m_currentVelocity.data.va );

		//Calculate GDWA
		CVelocity2D vel = gdwa->calculateGDWA();

		//Output velocity
		m_velocity.data.vx = vel.getV();
		m_velocity.data.va = vel.getW();
		m_velocityOut.write();

		cout<<"V::"<<m_velocity.data.vx<<"  W::"<<m_velocity.data.va<<endl;

	}
	else if( curTime-preTime > 5 )
	{
		m_velocity.data.vx = 0;
		m_velocity.data.va = 0;
		m_velocityOut.write();
		cout<<"V::"<<m_velocity.data.vx<<"  W::"<<m_velocity.data.va<<endl;
	}

  return RTC::RTC_OK;
}

/*
RTC::ReturnCode_t RVC2011::onAborting(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/

/*
RTC::ReturnCode_t RVC2011::onError(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/

/*
RTC::ReturnCode_t RVC2011::onReset(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/

/*
RTC::ReturnCode_t RVC2011::onStateUpdate(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/

/*
RTC::ReturnCode_t RVC2011::onRateChanged(RTC::UniqueId ec_id)
{
  return RTC::RTC_OK;
}
*/



extern "C"
{
 
  void RVC2011Init(RTC::Manager* manager)
  {
    coil::Properties profile(rvc2011_spec);
    manager->registerFactory(profile,
                             RTC::Create<RVC2011>,
                             RTC::Delete<RVC2011>);
  }
  
};


