/*****************************************************************************************************
RVC2011 RT-Component
Copyright (c) 2011, Kyoto Univ.
All rights reserved.

Contact us if you use this software for sell.
If you use this software not for sell, you can use this software under BSD lisence.
See the files LICENSE.TXT and LICENSE-BSD.TXT for more details.                     
*****************************************************************************************************/
/*+---------------------------------------------------------------------------+
  |   Programming Tools for Robot Control in MatsunoLab(PTRC).      	      |
  |   copyright @ Kazuyuki Kon 											      |
  +---------------------------------------------------------------------------+*/
/*!
	@file CPoint2D.h
	@brief 
	@author Kazuyuki Kon, Kyoto University
*/

#ifndef CPOINT2D_H
#define CPOINT2D_H

#include "CPointBase.h"


#include <iostream>

namespace ptrc
{
	namespace poses
	{
		/*---------------------------------------------------------------*/
		/*!
		*	@class struct to represent robot's pose in SU(2)
		*/
		/*---------------------------------------------------------------*/
		class CPoint2D : public CPointBase
		{
		public:
			/*!
			*	@brief constructor
			*/
			CPoint2D(void)
			{
				m_x = 0;
				m_y = 0;
			}
			
			/*!
			*	@brief copy constructor
			*	@param x0
			*	@param y0
			*/
			CPoint2D(double x0,double y0)
			{
				m_x = x0;
				m_y = y0;
			}

			/*!
			*	@brief destructor
			*/
			~CPoint2D(void){};

			/*!
			*	@brief Overload of Operator=
			*	@param point 
			*	@retval 
			*/
			CPoint2D& operator=(const CPoint2D& point)
			{
				this->m_time = point.m_time;
				this->m_x   = point.m_x;
				this->m_y   = point.m_y;
				return *this;
			}

			/*!
			*	@brief Overload of Operator+
			*	@param point 
			*	@retval
			*/
			CPoint2D operator+(CPoint2D point)
			{
				CPoint2D temp;
				temp.m_x = m_x + point.getX();
				temp.m_y = m_y + point.getY();
				return temp;
			}
			
			/*!
			*	@brief Overload of Operator-
			*	@param point 
			*	@retval
			*/
			CPoint2D operator-(CPoint2D point)
			{
				CPoint2D temp;
				temp.m_x = m_x - point.getX();
				temp.m_y = m_y - point.getY();
				return temp;
			}

			/*!
			*	@brief Overload of Operator[]
			*	@param i index indicated as 0:mx,1:my,2:time
			*	@retval mx(or my, or time)
			*/
            virtual double &operator[](unsigned int i)
            {
		        switch(i)
                {
			        case 0:return m_x;
				    case 1:return m_y;
					case 2:return m_time;
                    default:
						throw std::runtime_error("CPoint2D::operator[]: Index of bounds.");
                 }
            }


			/*!
			*	@brief Calculate the distance between this and pos2
			*/
			double dist(CPoint2D pos2)
			{
				return sqrt( pow((getX()-pos2.getX()),2.0) + pow((getY()-pos2.getY()),2.0) );
			};


			/*!
			*	@brief set X data
			*	@param[in] x0 value of X coordinate
			*/
			void setX(double x0){m_x = x0;};

			/*!
			*	@brief set Y data
			*	@param[in] y0 value of Y coordinate
			*/
			void setY(double y0){m_y = y0;};

			/*!
			*	@brief set XY data
			*	@param[in] x0 value of X coordinate
			*	@param[in] y0 value of Y coordinate
			*/
			void setXY(double x0,double y0)
			{
				m_x = x0;
				m_y = y0;
			};

			/*!
			*	@brief get X
			*	@retval 
			*/
			double getX(){return m_x;};

			/*!
			*	@brief get Y
			*	@retval 
			*/
			double getY(){return m_y;};


			/*!
			*	@brief print registed data
			*/
			virtual void printData()
			{
				std::cout<<m_time<<" "<<m_x<<" "<<m_y<<std::endl;
			}

		protected:
			double m_x;			//!< x[m]
			double m_y;			//!< y[m]
		};


	}
}

#endif
