/*****************************************************************************************************
RVC2011 RT-Component
Copyright (c) 2011, Kyoto Univ.
All rights reserved.

Contact us if you use this software for sell.
If you use this software not for sell, you can use this software under BSD lisence.
See the files LICENSE.TXT and LICENSE-BSD.TXT for more details.                     
*****************************************************************************************************/
/*+---------------------------------------------------------------------------+
  |   Programming Tools for Robot Control in MatsunoLab(PTRC).      	      |
  |   copyright @ Kazuyuki Kon 											      |
  +---------------------------------------------------------------------------+*/
/*!
	@file CPose2D.h
	@brief 
	@author Kazuyuki Kon, Kyoto University
*/

#ifndef CPOSE2D_H
#define CPOSE2D_H

#include "CPoint2D.h"

namespace ptrc
{
	namespace poses
	{ 
		/*---------------------------------------------------------------*/
		/*!
		*	@class 
		*/
		/*---------------------------------------------------------------*/
		class CPose2D : public CPoint2D
		{
		public:
			/*!
			*	@brief destructor
			*/
			CPose2D(void)
			{
				m_yaw = 0;
			}

			/*!
			*	@brief destructor
			*/
			CPose2D(double x0, double y0, double yaw0)
			{
				m_x   = x0;
				m_y   = y0;
				m_yaw = yaw0;
			}

			/*!
			*	@brief destructor
			*/
			~CPose2D(void){};

			/*!
			*	@brief Overload of Operator=
			*	@param point 
			*	@retval 
			*/
			CPose2D& operator=(const CPose2D& pose)
			{
				m_time = pose.m_time;
				m_x    = pose.m_x;
				m_y    = pose.m_y;
				m_yaw  = pose.m_yaw; 
				return *this;
			}

			/*!
			*	@brief Overload of Operator+
			*	@param point 
			*	@retval
			*/
			CPose2D operator+(CPose2D pose)
			{
				CPose2D temp;
				temp.m_x   = m_x + pose.getX();
				temp.m_y   = m_y + pose.getY();
				temp.m_yaw = m_yaw + pose.getYaw();
				return temp;
			}
			
			/*!
			*	@brief Overload of Operator-
			*	@param point 
			*	@retval
			*/
			CPose2D operator-(CPose2D pose)
			{
				CPose2D temp;
				temp.m_x   = m_x - pose.getX();
				temp.m_y   = m_y - pose.getY();
				temp.m_yaw = m_yaw - pose.getYaw();
				return temp;
			}
			
			/*!
			*	@brief Overload of Operator[]
			*	@param i index indicated as 0:mx,1:my,2:time
			*	@retval mx(or my, or time)
			*/
            double &operator[](unsigned int i)
            {
		        switch(i)
                {
			        case 0:return m_x;
				    case 1:return m_y;
					case 2:return m_yaw;
					case 3:return m_time;
                    default:
						throw std::runtime_error("CPose2D::operator[]: Index of bounds.");
                 }
            }
			
			/*!
			*	@brief [NbhvZ 
			*/
			double dist(CPose2D pos2)
			{
				return sqrt( pow((getX()-pos2.getX()),2.0) + pow((getY()-pos2.getY()),2.0) );
			};

			/*!
			*	@brief set yaw angle data
			*	@param yaw0 
			*/
			void setYaw(double yaw0){m_yaw = yaw0;};

			/*!
			*	@brief set x,y,yaw data
			*	@param x0
			*	@param y0
			*	@param yaw0 
			*/
			void setXYYaw(double x0,double y0,double yaw0)
			{
				m_x = x0;
				m_y = y0;
				m_yaw = yaw0;
			}

			/*!
			*	@brief get yaw angle data
			*	@retval yaw angle 
			*/
			double getYaw(){return m_yaw;};

			/*!
			*	@brief print registed data
			*/
			virtual void printData()
			{
				std::cout<<m_time<<" "<<m_x<<" "<<m_y<<" "<<m_yaw<<std::endl;
			}

			/*!
			*	@brief CPose2DCPoint2D݂̂̕𒊏oCo͂
			*/
			CPoint2D getPoint(void)
			{
				CPoint2D point;
				point.setTime(this->m_time);
				point.setXY(this->m_x,this->m_y);
				return point;
			}


		protected:
			double m_yaw;		//!< yaw angle[rad]

		};



	}
}

#endif