/*****************************************************************************************************
RVC2011 RT-Component
Copyright (c) 2011, Kyoto Univ.
All rights reserved.

Contact us if you use this software for sell.
If you use this software not for sell, you can use this software under BSD lisence.
See the files LICENSE.TXT and LICENSE-BSD.TXT for more details.                     
*****************************************************************************************************/
/*****************************************************************************************************/
/*!
*	@file GlobalDWA.cpp
*	@brief Class for trajectory generation based on GDWA
*	@author goto@uec, kon@kyoto-u
*/
/*****************************************************************************************************/
#include "GlobalDWA.h"

/*****************************************************************************************************/
//
/*****************************************************************************************************/
inline bool dsec(const trajectory& left, const trajectory& right)
{
  return left.obj > right.obj ;
}

/*****************************************************************************************************/
//Load map
/*****************************************************************************************************/
void GlobalDWA::loadMap(string name,int length)
{
	curMapNum++;
	flag_new_image = false;

	if(curMapNum>=2)
	{
		//Get global coordinate of the last way point in the previois map
		pre_g_target = g_target;
	}

	for(int i=1;i<length;i++)
	{
		if(name[i-1]=='0' && name[i]=='0')
		{
			char name_num[3];
			sprintf(name_num,"%d%d",curMapNum/10,curMapNum%10);
			name[i-1]=name_num[0];
			name[i] = name_num[1];
			break;
		}
	}
	string sname = name;
	bmp.Load(sname);  //Load bitmap map

	cout<<"Load::"<<sname.c_str()<<endl;

	//Get color information of each cell
	bp.gridnum.Width  = bmp.GetWidth();  //Number of pixel corresponds to the number of grids
	bp.gridnum.Height = bmp.GetHeight(); //Number of pixel corresponds to the number of grids

	if(curMapNum >= 1)
	{
		bp.gridsize.W = GRIDSIZE;//
		bp.gridsize.H = GRIDSIZE;//
	}

	//Size of map
	bp.lgridnum.Width  = LOCALSIZE_W;
	bp.lgridnum.Height = LOCALSIZE_H;
	bp.lgridsize.W     = bp.gridsize.W;//[m]
	bp.lgridsize.H     = bp.gridsize.H;//[m]

	//==Get information about waypoints, obstacles and RVCs====//
	pixelPos clear = {0,0};
	int number[256];
	memset(number,0,sizeof(number));

	int nn=0;//Total number of waypoints
	for(int i=0;i<256;i++) bp.waypoint[i] = clear;
	bp.waypoint_num_all = 0;

	for(int i=0; i<bp.gridnum.Width; i++)
	{
		for(int j=0; j<bp.gridnum.Height; j++)
		{
			COLOR24 lcolor = bmp.GetPixel(i,j);
			//Check the waypoints
			if(lcolor.r != 0 && lcolor.g == 0 && lcolor.b != 0)
			{
				number[nn] = (int)lcolor.r;
				bp.waypoint[nn].Width = bp.gridnum.Width-i;//right bottom side of the image should be origin
				bp.waypoint[nn].Height= j;
				nn++;
			}
			//Check the RVCs
			int blue = (int)lcolor.b;
			cell[i][j].const_vel = robotSpec.vmax * floor((double)blue/255.0*10.0)/10.0;//Color  corresponds to the velocity constraints
			
			//Check NF
			if(cell[i][j].const_vel == 0)
			{
				cell[i][j].NF.center = OBS_NF;
			}
			else
			{
				cell[i][j].NF.center = INI_NF;
			}
		}//for j
	}//for i
	bp.waypoint_num_all = nn;

	//========Sort the waypoints=====
	bool flagnum = false;
	while( !flagnum )
	{
		flagnum = true;
		for( int i=1; i<nn; i++ )
		{
			if( number[i] > number[i-1] )
			{
				int temp         = number[i-1];
				number[i-1]      = number[i];
				number[i]        = temp;
				pixelPos temp2   = bp.waypoint[i-1];//ith waypoint, jth waypoint
				bp.waypoint[i-1] = bp.waypoint[i];
				bp.waypoint[i]   = temp2;
				flagnum = false;
			}
		}
	}

	//Calculate the anglar difference between global coordinate and local coordinate
	//bp.waypoint[0].Width means (i,j) of cell, so has to be changed
	bp.waypoint_target = 1;
	if( curMapNum == 1 )
	{
		double diffx = (double)(bp.waypoint[1].Height-bp.waypoint[0].Height)*bp.gridsize.H;
		double diffy = (double)(bp.waypoint[1].Width -bp.waypoint[0].Width )*bp.gridsize.W;
		offset.setYaw( atan2(diffy,diffx) );
		cout<<"offsettheta::"<<offset.getYaw()*180/M_PI<<endl;
	}

	//
	offset.setX( (double)bp.waypoint[0].Height*bp.gridsize.H + bp.gridsize.H/2 );//(Perhaps)Offset from the right bottom of image
	offset.setY( (double)bp.waypoint[0].Width *bp.gridsize.W + bp.gridsize.W/2 );

	//Show to console
	cout<<"bp.waypoint_num_all:"<<bp.waypoint_num_all<<endl;
	for(int i=0;i<bp.waypoint_num_all;i++)
	{
		cout<<"waypoint No."<<i<<" w:"<<bp.waypoint[i].Width<<" h:"<<bp.waypoint[i].Height<<endl;
	}

	return;
}

/*****************************************************************************************************/
//Initialize parameters
/*****************************************************************************************************/
bool GlobalDWA::initializeParameters(void)
{
	//Conver to Global coordinate
	g_odo.setXYYaw(0,0,0);
	offset.setXYYaw(0,0,0);
	g_target.setXY(0,0); 
	pre_g_target.setXY(0,0);
	return true;
}

/*****************************************************************************************************/
//Calculate position of the robot in Global coordinate
/*****************************************************************************************************/
void GlobalDWA::calculatePositionGL(double _x, double _y, double _theta )
{
		g_odo = convertFrameGLtoMapGL(CPose2D(_x,_y,_theta));
}

/*****************************************************************************************************/
//Set obstacle information
/*****************************************************************************************************/
void GlobalDWA::setObst(double obst0, double theta0)
{
	CPoint2D obstPosition;
	obstPosition.setX( obst0 * cos(theta0) );
	obstPosition.setY( obst0 * sin(theta0) );

	//Get ID from x,y coordinate in local frame
	pixelPos id = getID(obstPosition);
	if(id.Width<0 || id.Width>LOCALSIZE_W || id.Height<0 || id.Height>LOCALSIZE_H )
	{
		cout<<"LocalMap"<<endl;
	}
	local[id.Height][id.Width].NF.center = OBS_NF;

	//Calculate the distance between nearest obstacle
	CPoint2D xy = getXY_L(id);
	double dist = sqrt(pow(xy.getX(),2.0)+pow(xy.getY(),2.0));
	if( min_dist > dist )
	{
		min_dist = dist;
	}
}

/*****************************************************************************************************/
//Set Obstacle information
/*****************************************************************************************************/
void GlobalDWA::setObstacle(double _x, double _y)
{
	CPoint2D  obstPosition(_x, _y);
	pixelPos id = getID(obstPosition);

	if(id.Width<0 || id.Width>LOCALSIZE_W || id.Height<0 || id.Height>LOCALSIZE_H )
	{
		cout<<"Out of LocalMap"<<endl;
	}
	local[id.Height][id.Width].NF.center = OBS_NF;

	CPoint2D xy = getXY_L(id);
	double dist = sqrt(pow(xy.getX(),2.0)+pow(xy.getY(),2.0));

	if( min_dist>dist )min_dist = dist;
}

/*****************************************************************************************************/
//Check way point 
/*****************************************************************************************************/
bool GlobalDWA::checkWaypoint(void)
{
	g_target.setX( (double)(bp.waypoint[bp.waypoint_target].Height-bp.waypoint[0].Height)*bp.gridsize.H + pre_g_target.getX() );
	g_target.setY( (double)(bp.waypoint[bp.waypoint_target].Width -bp.waypoint[0].Width )*bp.gridsize.W + pre_g_target.getY() );
	
	cout<<"tgx:"<<g_target.getX()<<" tgy:"<<g_target.getY()<<" gx:"<<g_odo.getX()<<" gy:"<<g_odo.getY()<<endl;
	
	if(sqrt(pow(g_target.getX()-g_odo.getX(),2)+pow(g_target.getY()-g_odo.getY(),2)) <= margin)
	{
		bp.waypoint_target++;
		cout<<"waypoint No::"<<bp.waypoint_target<<endl;
		if(bp.waypoint_target==bp.waypoint_num_all)
		{
			flag_new_image = true;
		}
	}
	return true;
}

/*****************************************************************************************************/
//
/*****************************************************************************************************/
double GlobalDWA::PItoPI(double theta0)
{
	while(theta0<=-M_PI){ theta0 = theta0 + 2*M_PI;}
	while(theta0 > M_PI){ theta0 = theta0 - 2*M_PI;}
	return theta0;
}

/*****************************************************************************************************/
//
/*****************************************************************************************************/
pixelPos GlobalDWA::getID(CPoint2D pos)
{
	pixelPos id;
	if( isValid(pos.getX(),pos.getY()) )//Check wether the robot is within the course 
	{
		id.Height = (int)(floor( (pos.getX()+(double)bp.lgridnum.Height*bp.lgridsize.H/2)/bp.lgridsize.H ));
		id.Width  = (int)(floor( (pos.getY()+(double)bp.lgridnum.Width *bp.lgridsize.W/2)/bp.lgridsize.W ));
	}else
	{
		id.Height=-1;id.Width=-1;
	}
	return id;
}

/*****************************************************************************************/
//あってる？？？？ in local Map
/*****************************************************************************************/
inline double GlobalDWA::getXFromPixel(int idh0)
{
	return (double)idh0*bp.lgridsize.H + bp.lgridsize.H/2 - bp.lgridsize.H*(double)bp.lgridnum.Height/2;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
inline double GlobalDWA::getYFromPixel(int idw0)//
{
	return (double)idw0*bp.lgridsize.W + bp.lgridsize.W/2 - bp.lgridsize.W*(double)bp.lgridnum.Width/2;
}

/*****************************************************************************************/
//Check the point(x,y) is on the grid map.
/*****************************************************************************************/
inline bool GlobalDWA::isValid(double x,double y)//
{
	if( (fabs(y) > (double)bp.lgridnum.Width*bp.lgridsize.W/2) || (fabs(x) > (double)bp.lgridnum.Height*bp.lgridsize.H/2) )//�R�[�X�O
	{
		return false;
	}
	else
	{
		return true;
	}
}

/*****************************************************************************************/
//Convert ?? into local map
/*****************************************************************************************/
void GlobalDWA::transLocal(void)
{
	//Conver xy coordinate in local frame into global frame, then return the corresponding global cell information 
	for( int idh=0; idh<LOCALSIZE_H; idh++ )
	{
		for( int idw=0; idw<LOCALSIZE_W; idw++ )
		{
			if( local[idh][idw].NF.center == INI_NF )
			{
				double lx = getXFromPixel(idh);
				double ly = getYFromPixel(idw);

				double tx = lx*cos(g_odo.getYaw()) - ly*sin(g_odo.getYaw()) + g_odo.getX() - pre_g_target.getX() + offset.getX();//transform x
				double ty = lx*sin(g_odo.getYaw()) + ly*cos(g_odo.getYaw()) + g_odo.getY() - pre_g_target.getY() + offset.getY();//transform y
			
				local[idh][idw].NF.center = getNFGlobal(tx,ty);//xy in local into tx,ty in global,then substitute them into localmap
				local[idh][idw].const_vel = getConstVelGlobal(tx,ty);
				
				//Revise the distance to nearest obstacle
				pixelPos id;
				id.Height   = idh;
				id.Width    = idw;
				CPoint2D xy = getXY_L(id);
				double dist = sqrt(pow(xy.getX(),2.0)+pow(xy.getY(),2.0));
				
				if(min_dist > dist)
				{
					min_dist = dist;
				}
			}
		}
	}
	return;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
bool GlobalDWA::initializeNF(void)
{
	for(int h=0;h<LOCALSIZE_H;h++)
	{
		for(int w=0;w<LOCALSIZE_W;w++)
		{
			local[h][w].const_vel = 0;
			local[h][w].grad      = 0;
			local[h][w].NF.center = INI_NF;
		}//for j
	}//for i

	min_dist = 998.0;
	return true;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
bool GlobalDWA::calculateManhattanDistance(void)
{
	//GDWA based on localmap
	int grid_x  = bp.lgridnum.Height;
	int grid_y  = bp.lgridnum.Width; 
	bool flag1  = true;
	double nf   = 0;
	double Man  = 0;
	double dMan = 1;//Additional cost to move to next grid

	//Convert ID of waypoint in global frame to ID in local frame.
	pixelPos refer_id =	getTargetPosition();

	//Substitute '0' to goal position
	if(refer_id.Height!=-1 || refer_id.Width!=-1)
	{
		local[refer_id.Height][refer_id.Width].NF.center = 0;
	}
	else
	{
		cout<<"Reference point is out of local map(Out of range)"<<endl;
		return false;
	}
	int pm_h[4] = {-1,1,0,0};//left,right,bottom,top
	int pm_w[4] = {0,0,-1,1};//left,right,bottom,top

	while(flag1)
	{
		flag1 = false;
		double min_nf = OBS_NF;
		for(int h=0;h<grid_x;h++)
		{
			for(int w=0;w<grid_y;w++)
			{
				if(local[h][w].NF.center == nf)
				{
					for(int cnt=0;cnt<4;cnt++)
					{
						//Subsutitute the value of next cell
						if(h+pm_h[cnt]>=0 && h+pm_h[cnt]<grid_x && w+pm_w[cnt]>=0 && w+pm_w[cnt]<grid_y)
						{
							if(local[h+pm_h[cnt]][w+pm_w[cnt]].NF.center == INI_NF){
								Man = dMan * robotSpec.vmax/local[h+pm_h[cnt]][w+pm_w[cnt]].const_vel;
								local[h+pm_h[cnt]][w+pm_w[cnt]].NF.center = nf + Man;

								if(min_nf > local[h+pm_h[cnt]][w+pm_w[cnt]].NF.center)
								{
									min_nf = local[h+pm_h[cnt]][w+pm_w[cnt]].NF.center;
								}
							}
						}
					}
				}//if
				if(local[h][w].NF.center>nf && local[h][w].NF.center<min_nf){//Find next candidate
					min_nf = local[h][w].NF.center;
				}else if(local[h][w].NF.center == INI_NF && !flag1){
					flag1 = true;
				}
			}//for j
		}//fori
		nf = min_nf;

		if( nf==OBS_NF )//!This line should be considered again!
		{
			break;
		}
	}//while


	//copy cell value
	for(int h=0;h<grid_x;h++)
	{
		for(int w=0;w<grid_y;w++)
		{
			//Lower
			if(h-1>=0){
				local[h][w].NF.lower = local[h-1][w].NF.center;
			}else{
				local[h][w].NF.lower = OBS_NF;
			}
			//Upper
			if(h+1<=grid_x){
				local[h][w].NF.upper = local[h+1][w].NF.center;
			}else{
				local[h][w].NF.upper = OBS_NF;
			}
			//Left
			if(w-1>=0){
				local[h][w].NF.left = local[h][w-1].NF.center;
			}else{
				local[h][w].NF.left = OBS_NF;
			}
			//Right
			if(w+1<=grid_y){
				local[h][w].NF.right = local[h][w+1].NF.center;
			}else{
				local[h][w].NF.right = OBS_NF;
			}
		}//for j
	}//for i
	return true;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
pixelPos GlobalDWA::getTargetPosition(void)
{
	//Convert ID of waypoint in global to local frame
	double g_target_x = (double)(bp.waypoint[bp.waypoint_target].Height-bp.waypoint[0].Height)*bp.gridsize.H + pre_g_target.getX();
	double g_target_y = (double)(bp.waypoint[bp.waypoint_target].Width -bp.waypoint[0].Width )*bp.gridsize.W + pre_g_target.getY();

	CPose2D refer_l;
	refer_l.setX(  (g_target_x-g_odo.getX())*cos(g_odo.getYaw()) + (g_target_y-g_odo.getY())*sin(g_odo.getYaw()) );
	refer_l.setY( -(g_target_x-g_odo.getX())*sin(g_odo.getYaw()) + (g_target_y-g_odo.getY())*cos(g_odo.getYaw()) );

	
	pixelPos refer_id = getID(refer_l);

	return refer_id;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
bool GlobalDWA::calculateNFGradient(void)
{
	int grid_x = bp.lgridnum.Height;
	int grid_y = bp.lgridnum.Width; 
	int pm_i[8] = {1,1,0,-1,-1,-1,0,1};
	int pm_j[8] = {0,1,1,1,0,-1,-1,-1};

	pixelPos target_id = getTargetPosition();
	double tx = getXFromPixel(target_id.Height);
	double ty = getYFromPixel(target_id.Width );
	double goal_theta = atan2(ty,tx);

	for(int i=0;i<grid_x;i++)
	{
		for(int j=0;j<grid_y;j++)
		{
			int curNF = local[i][j].NF.center;
			local[i][j].grad = 0;
			if(curNF==0 || curNF==OBS_NF)
			{
			}
			else
			{
				double gradiation = 0;
				for(int cnt=0;cnt<8;cnt++)
				{
					if(i+pm_i[cnt]>=0 && i+pm_i[cnt]<grid_x && j+pm_j[cnt]>=0 && j+pm_j[cnt]<grid_y)
					{
						double getNF = local[i+pm_i[cnt]][j+pm_j[cnt]].NF.center;
						
						if(curNF>getNF)
						{
							gradiation = cnt*M_PI/4;
							curNF = getNF;
						}else if(curNF==getNF)
						{
							if( fabs(goal_theta-cnt*M_PI/4) <= fabs(goal_theta-gradiation) )
							{
								gradiation = cnt * M_PI/4;
							}
						}
					}
				}//for cnt
				while(gradiation>  M_PI) gradiation = gradiation - 2*M_PI;
				while(gradiation<=-M_PI) gradiation = gradiation + 2*M_PI;
				local[i][j].grad = gradiation;
			}
		}//for j
	}//for i
	return true;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
CVelocity2D GlobalDWA::calculateGDWA(void)
{
	CVelocity2D velOut;

	//Navigation Function
	if( !calculateManhattanDistance() )
	{
		cout<<"error in Calclation of Navigation Function"<<endl;
		return velOut;
	}

	//Derive the grafient of NF
	if( !calculateNFGradient() )
	{
		cout<<"error in Calclation of Navigation Function gradient"<<endl;
		return velOut;
	}

	//Calculate trajectory
	velOut = trajectoryPlanning();

	return velOut;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
inline bool GlobalDWA::isCourseOutLO(pixelPos curpixel0){//
	if( curpixel0.Width<0 || curpixel0.Height<0 || 
		curpixel0.Width>bp.lgridnum.Width || curpixel0.Height>bp.lgridnum.Height)
	{
		return true;
	}
	return false;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
inline bool GlobalDWA::isCourseOutGL(pixelPos curpixel0)
{

	if(curpixel0.Width<0 || curpixel0.Height<0 
		|| curpixel0.Width>bp.gridnum.Width-1 || curpixel0.Height>bp.gridnum.Height-1)
	{
		return true;
	}
	return false;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
inline double GlobalDWA::getNFLocal(double x,double y)
{
	pixelPos curpixel = GetCellNum_L(x,y);
	double curNF = 0;
	if( isCourseOutLO(curpixel) )//If out of map, then it is considered as obstacle
	{
		curNF = OBS_NF;
	}else{
		curNF = local[curpixel.Height][curpixel.Width].NF.center;
	}
	return curNF;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
inline double GlobalDWA::getNFGlobal(double x,double y)
{
	pixelPos curpixel = GetCellNum_G(x,y);
	double curNF = 0;
	if( isCourseOutGL(curpixel) )//If out of map, then it is considered as obstacle
	{
		curNF = OBS_NF;
	}else{
		curNF = cell[curpixel.Width][curpixel.Height].NF.center;
	}
	return curNF;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
double GlobalDWA::getConstVelLocal(double x,double y)
{
	pixelPos curpixel = GetCellNum_L(x,y);
	double constvel;
	if( isCourseOutLO(curpixel) )//If out of map, then it is considered as obstacle
	{
		constvel = 0;
	}else{
		constvel = local[curpixel.Height][curpixel.Width].const_vel;
	}
	return constvel;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
double GlobalDWA::getConstVelGlobal(double x,double y)
{
	pixelPos curpixel = GetCellNum_G(x,y);
	double constvel;
	if( isCourseOutGL(curpixel) )
	{
		constvel = 0;
	}else{
		constvel = cell[curpixel.Width][curpixel.Height].const_vel;
	}
	return constvel;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
aroundNF GlobalDWA::GetNFall(double x,double y)
{
	pixelPos curpixel = GetCellNum_L(x,y);
	aroundNF nf;
	if( isCourseOutLO(curpixel) )//If out of map, then it is considered as obstacle
	{
		nf.center = OBS_NF;
		nf.left = OBS_NF;
		nf.right = OBS_NF;
		nf.upper = OBS_NF;
		nf.lower = OBS_NF;
	}else{
		nf = local[curpixel.Height][curpixel.Width].NF;
	}
	return nf;
}

/*****************************************************************************************/
//
/*****************************************************************************************/
double GlobalDWA::getGradient(double x,double y)
{
	pixelPos curpixel = GetCellNum_L(x,y);
	double gradiation = 0;
	if( isCourseOutLO(curpixel) )
	{
		gradiation = 0;
	}else{
		gradiation = local[curpixel.Height][curpixel.Width].grad;
	}
	return gradiation;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
inline pixelPos GlobalDWA::GetCellNum_L(double x,double y)
{
	pixelPos lcurpixel;

	//Origin is the bottom left side of image. x axis is set toward the top side. 
	//y axis is set toward the left side.
	lcurpixel.Width = (int)floor((y+(double)bp.lgridnum.Width*bp.lgridsize.W/2)/bp.lgridsize.W);//Check the coordinate of W and H!!
	lcurpixel.Height= (int)floor((x+(double)bp.lgridnum.Height*bp.lgridsize.H/2)/bp.lgridsize.H);
	return lcurpixel;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
inline pixelPos GlobalDWA::GetCellNum_G(double x,double y)
{
	pixelPos lcurpixel;
	lcurpixel.Width = bp.gridnum.Width-(int)(y/bp.gridsize.W)-1;//
	lcurpixel.Height = (int)(x/bp.gridsize.H);					//
	return lcurpixel;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
inline CPoint2D GlobalDWA::getXY(pixelPos W_H)
{
	CPoint2D xy;
	xy.setX( (double)W_H.Height*bp.gridsize.H + bp.gridsize.H/2 );
	xy.setY( (double)(bp.gridnum.Width-W_H.Width-1)*bp.gridsize.W + bp.gridsize.W/2);
	return xy;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
inline CPoint2D GlobalDWA::getXY_L(pixelPos W_H)
{
	CPoint2D xy;
	xy.setX( (double)W_H.Height*bp.lgridsize.H+bp.lgridsize.H/2  - (double)bp.lgridnum.Height*bp.lgridsize.H/2 );
	xy.setY( -((double)W_H.Width *bp.lgridsize.W+bp.lgridsize.W/2) + (double)bp.lgridnum.Width *bp.lgridsize.W/2);
	return xy;
}
/*****************************************************************************************/
//
/*****************************************************************************************/
inline int GlobalDWA::getWayPointNum(double x,double y,int cur_num)
{
	if(sqrt(pow(targetxy.getX()-x,2)+pow(targetxy.getY()-y,2)) <= margin)
	{
		cur_num = cur_num + 1;
	}
	return cur_num;
}

/*****************************************************************************************/
//Execute the trajectory generation
/*****************************************************************************************/
CVelocity2D GlobalDWA::trajectoryPlanning(void)
{
	CVelocity2D velOut;

	std::vector<trajectory> LOG;
	trajectory start_point;
	start_point.poseLocal.setXY(0,0);
	start_point.obj = 0;
	start_point.v   = velPre;
	LOG.push_back( start_point );

	double DWv[5], DWw[9];//
	int DWvcnt=0, DWwcnt=0;

	int real_n = 1;	//number of candidates to be left to next T

	std::vector<trajectory> DATA;

	pixelPos goal_id = getTargetPosition();
	double goal_x = getXFromPixel(goal_id.Height);
	double goal_y = getYFromPixel(goal_id.Width );

	cout<<"Local  goalx:"<<goal_x<<" goaly:"<<goal_y<<endl;
	cout<<"waypoint map:"<<curMapNum<<" waypoint:"<<bp.waypoint_target<<"/"<<bp.waypoint_num_all<<endl;

	for(int horizon=0; horizon<horizon_num; horizon++)
	{
		DATA.clear();
		for(int tra=0; tra<real_n; tra++)
		{
			//Select acc 
			if(LOG[tra].v.getV() == 0){ DWv[0]=robotSpec.avmax/2; DWv[1]=robotSpec.avmax; DWvcnt=2;}
			else if(LOG[tra].v.getV() >= robotSpec.vmax){ DWv[0]=-robotSpec.avmax; DWv[1] =-robotSpec.avmax/2; DWv[2]=0; DWvcnt=3;}
			else{ DWv[0]=-robotSpec.avmax; DWv[1]=-robotSpec.avmax/2; DWv[2]=0; DWv[3]=robotSpec.avmax/2; DWv[4]=robotSpec.avmax; DWvcnt=5;}

			if(LOG[tra].v.getW() >= robotSpec.wmax){ 
				DWw[0]=0; DWw[1]=-robotSpec.awmax/3; DWw[2]=-2*robotSpec.awmax/3; DWw[3]=-robotSpec.awmax; DWwcnt=4;
			}
			else if(LOG[tra].v.getW() <= -robotSpec.wmax){ DWw[0]=0; DWw[1]= robotSpec.awmax/3;DWw[2]= 2*robotSpec.awmax/3; DWw[3]= robotSpec.awmax; DWwcnt=4;}
			else{ DWw[0]=-robotSpec.awmax; DWw[1]=-2*robotSpec.awmax/3; DWw[2]=-robotSpec.awmax/3;DWw[3]=0; 
                               DWw[4]=robotSpec.awmax/3; DWw[5]=2*robotSpec.awmax/3; DWw[6]=robotSpec.awmax; DWwcnt=7;}
	
			std::vector<acceralation> accs;
			for(int i=0;i<DWvcnt;i++)
			{
				for(int j=0;j<DWwcnt;j++)
				{
					accs.push_back(acceralation(DWv[i], DWw[j]));
				}
			}

			double curNF = dNF1( LOG[tra].poseLocal.getX(), LOG[tra].poseLocal.getY() );

			for(int i=0;i<accs.size();i++)
			{
				double v     = LOG[tra].v.getV();
				double w     = LOG[tra].v.getW();
				double acc_v = accs[i].av;
				double acc_w = accs[i].aw;
				double xl    = LOG[tra].poseLocal.getX();
				double yl    = LOG[tra].poseLocal.getY();
				double thetal= LOG[tra].poseLocal.getYaw();
				double simu_ts = robotSpec.tsamp;
				double NF=0, dNF=0, VEL=0, GOAL=0;
				double v0=0,w0=0;
				double v00=0,w00=0;
				double min_dist_l = 40.0;
				bool   flagout    = false;

				for(int dt=0; dt<(int)(T/simu_ts);dt++)
				{
					//Velocity after sim_ts
					v = v + acc_v*simu_ts;
					w = w + acc_w*simu_ts;
					if(v>robotSpec.vmax) v = robotSpec.vmax;
					else if(v < 0)	v = 0;
					if(w>robotSpec.wmax)	w = robotSpec.wmax;
					else if(w < -robotSpec.wmax) w = -robotSpec.wmax;
					
					//Local planning trajectory
					xl +=  (v*simu_ts)*cos(thetal + (w*simu_ts)/2);
					yl +=  (v*simu_ts)*sin(thetal + (w*simu_ts)/2);
					thetal = thetal + w*simu_ts;

					//admissible velocity in this place
					double min_const_vel = getConstVelLocal(xl,yl);

					//Calculate ad region
					min_const_vel = calculateADRegion(xl,yl,min_const_vel);

					//Velocity after simu_ts
					if( v > min_const_vel )
					{
						xl -=  (v*simu_ts)*cos(thetal + (w*simu_ts)/2);
						yl -=  (v*simu_ts)*sin(thetal + (w*simu_ts)/2);
						thetal = thetal - w*simu_ts;

						//keep radious of circle
						w = w *( min_const_vel / v );
						v = min_const_vel;

						xl +=  (v*simu_ts)*cos(thetal + (w*simu_ts)/2);
						yl +=  (v*simu_ts)*sin(thetal + (w*simu_ts)/2);
						thetal = thetal + w*simu_ts;
	
					}
					//Calculate NF 
					double NFdata = getNFLocal(xl,yl);
					if(NFdata != OBS_NF)
					{
						for(double fai=0;fai<2*M_PI;fai+=M_PI/4){///Shortest Distance Shearching
							if(getNFLocal(xl+robotSpec.width/2*cos(fai)*0.9,yl+robotSpec.width/2*sin(fai)*0.9) == OBS_NF)
							{
								NFdata = OBS_NF;
								break;
							}
						}
					}
					if(NFdata == OBS_NF)
					{
						flagout = true;
						if(DATA.size()==0 && horizon==0)
						{
							double dist_l = sqrt(pow(xl,2.0)+pow(yl,2.0));
							if( min_dist_l > dist_l )
							{
								min_dist_l = dist_l;
								double theta_obs = atan2(yl,xl);
								if( theta_obs>0 || (theta_obs==0 && emergencyRotationDirection=='R') )
								{
									emergencyRotationDirection = 'R';
								}else{
									emergencyRotationDirection = 'L';
								}
							}
						}
						break;
					}
	
					if(dt==0 && horizon==0)
					{
						v00 = v;
						w00 = w;
					}

					//
					if( dt==0 )
					{
						if(horizon == 0)
						{
							v0 = v00 + acc_v*robotSpec.tsamp;
							w0 = w00 + acc_w*robotSpec.tsamp;
							if( v0 > v ) v0 = v;
							if(fabs(w0)>fabs(w)) w0 = w;
						}else{
							v0 = LOG[tra].v0.getV();
							w0 = LOG[tra].v0.getW();
						}
					}
				}//for dt

				if( !flagout )
				{
					//NF
					double diffangle = getGradient(xl,yl)-thetal;
					while(diffangle > M_PI) diffangle = diffangle - 2*M_PI;
					while(diffangle<=-M_PI) diffangle = diffangle + 2*M_PI;

					//Normalize NF with PI
					NF = ( M_PI - fabs(diffangle) )/M_PI;

					//GOAL
					double theta_G = atan2(goal_y-yl,goal_x-xl);
					double difftheta_G = fabs(theta_G - thetal);
					while(difftheta_G > M_PI) difftheta_G = fabs(difftheta_G - 2*M_PI);
					GOAL = fabs(M_PI - difftheta_G)/M_PI;

					//VEL
					VEL = v/robotSpec.vmax;

					//dNF1
					double predi_dNF1 = dNF1(xl,yl);
					dNF = (curNF - predi_dNF1)/2;
					if(dNF < 0)    dNF = 0.001;
					else if(dNF>1) dNF = 0.5;

					trajectory tmp;
					tmp.obj = a*NF + b*GOAL + c*VEL + d*dNF + LOG[tra].obj;
					tmp.poseLocal.setXYYaw(xl, yl, thetal);
					tmp.v.setVW(v ,w );
					tmp.v0.setVW(v0 ,w0 );
					DATA.push_back( tmp );
				}//if flagout
			}//for i
		}//for tra

		
		if( !DATA.empty() )
		{
			//Sort Data
			std::sort( DATA.begin(), DATA.end(), dsec );	
			emergencyCount =0;
		}
		else if(DATA.empty() && horizon==0)
		{
			emergencyCount++;
			if(emergencyCount > 2)
			{
				emergencyCount=1;
				if(emergencyRotationDirection=='R')
				{
					velOut.setVW(0, -robotSpec.wmax/2);
				}
				else
				{
					velOut.setVW(0, robotSpec.wmax/2);
				}
			}else{
				velOut.clear();
			}
			break;
		}
		if(DATA.size()>=n)
		{
			LOG.clear();
			for(int i=0;i<n;i++) LOG.push_back(DATA[i]);
			real_n = n;
		}
		else{
			LOG.clear();
			for(int i=0;i<DATA.size();i++) LOG.push_back(DATA[i]);
			real_n = DATA.size();
		}

		//OutPut
		velOut = LOG[0].v0;
	}//for horizon

	if(emergencyCount!=0){
		//printf("emergencyCount::%d LR:%c\n",emergencyCount,emergencyRotationDirection);
		printf("There is no valid predicted trajectory");
	}

	velPre = velOut;
	return  velOut;
}

/*****************************************************************************************/
//(x,y) is in local frame
/*****************************************************************************************/
double GlobalDWA::dNF1(double x,double y)
{
	pixelPos curpixel = GetCellNum_L(x,y);
	if( isCourseOutLO(curpixel) )
	{
		return OBS_NF;
	}
	else
	{
		aroundNF NF = GetNFall(x,y);
		if(NF.center == OBS_NF)
		{
			return NF.center;
		}
		CPoint2D x0y0 = getXY_L(curpixel);
		double x0 = x0y0.getX();
		double y0 = x0y0.getY();
		double diffx = x-x0;
		double diffy = y-y0;

		double ru=0,lu=0,rl=0,ll=0,x1=0,y1=0;
		if(diffx>=0 && diffy<0)
		{
			ru=((double)(NF.right+NF.upper))/2;
			lu=((double)(NF.upper+NF.center))/2;
			rl=((double)(NF.right+NF.center))/2;
			ll=(double)NF.center;
			x1=diffx;
			y1=bp.lgridsize.W/2-fabs(diffy);
		}
		else if(diffx>=0 && diffy>=0)
		{
			ru=((double)(NF.upper+NF.center))/2;
			lu=((double)(NF.left+NF.upper))/2;
			rl=(double)NF.center;
			ll=((double)(NF.left+NF.center))/2;
			x1=diffx;
			y1=diffy;
		}
		else if(diffx<0 && diffy>=0)
		{
			ru=(double)NF.center;
			lu=((double)(NF.left+NF.center))/2;
			rl=((double)(NF.lower+NF.center))/2;
			ll=((double)(NF.lower+NF.left))/2;
			x1=bp.lgridsize.H/2-fabs(diffx);
			y1=diffy;
		}
		else if(diffx<0 && diffy<0)
		{
			ru=((double)(NF.right+NF.center))/2;
			lu=(double)NF.center;
			rl=((double)(NF.right+NF.lower))/2;
			ll=((double)(NF.lower+NF.center))/2;
			x1=bp.lgridsize.H/2-fabs(diffx);
			y1=bp.lgridsize.W/2-fabs(diffy);
		}
		double curNF = (((lu-ll)/(bp.lgridsize.H/2)*x1+ll) - ((ru-rl)/(bp.lgridsize.H/2)*x1+rl))/(bp.lgridsize.W/2)*y1+((ru-rl)/(bp.lgridsize.H/2)*x1+rl);
		return curNF;
	}
}

/*****************************************************************************************/
//
/*****************************************************************************************/
//void GlobalDWA::CalcTargetPos(void)
//{
//	IIS::TimedPosition ltarget = getXY(bp.waypoint[bp.waypoint_target]);
//	IIS::TimedPosition preltarget = getXY(bp.waypoint[bp.waypoint_target-1]);
//	targetxy.x = targetxy.x + (ltarget.x-preltarget.x);
//	targetxy.y = targetxy.y + (ltarget.y-preltarget.y);
//}

/*****************************************************************************************/
//Calculate the AD region 
/*****************************************************************************************/
double GlobalDWA::calculateADRegion(double xl,double yl,double min_const_vel)
{
	//  48 around cells
	for(double h_cnt=-3;h_cnt<=3;h_cnt++)
	{
		for(double w_cnt=-3;w_cnt<=3;w_cnt++)
		{
			double const_v = getConstVelLocal(xl+h_cnt*bp.lgridsize.H,yl+w_cnt*bp.lgridsize.W);
			if(const_v < min_const_vel && !(h_cnt==0 && w_cnt==0) )
			{

				//
				pixelPos p = GetCellNum_L(xl+h_cnt*bp.lgridsize.H,yl+w_cnt*bp.lgridsize.W);
				CPoint2D shortest = getXY_L( p );

				if(h_cnt>0){ shortest.setX(shortest.getX() - bp.lgridsize.H/2); }
				else if(h_cnt<0){ shortest.setX(shortest.getX() + bp.lgridsize.H/2); }

				if(w_cnt>0){ shortest.setY(shortest.getY() - bp.lgridsize.W/2); }
				else if(w_cnt<0){ shortest.setY(shortest.getY() + bp.lgridsize.W/2); }

				double shortLj = sqrt( pow(shortest.getX()-xl,2)+pow(shortest.getY()-yl,2) );
				double V_Lj    = sqrt( pow(const_v,2)+2*shortLj*robotSpec.avmax);
 
				if(V_Lj < min_const_vel)
				{
					min_const_vel = V_Lj;
				}
			}//if
		}//for w_cnt
	}//for h_cnt
	return min_const_vel;
}

