/*
 * MasterControllerClient.cpp
 *
 *  Created on: 2016/09/17
 *      Author: vivitter
 */

#include "CMasterControllerClient.h"


CMasterControllerClient::CMasterControllerClient(char* deviceName, unsigned long baudRate) : CSerialPort(deviceName, baudRate) {
	// Initialize variables
	this->_remain = 0;
	this->_crcErrorCnt = 0;
	this->findHeader1 = false;
	this->findHeader2 = false;
}

CMasterControllerClient::~CMasterControllerClient() {
	this->disconnect();
}

/* ======================================
 * Private
 * ====================================== */
uint32_t CMasterControllerClient::searchPacket(uint8_t* readBytes, uint32_t length, DMA_RX_DATA* rxData)
{
	uint8_t buf[MPCOM_BUFF_SIZE];
	int32_t packetHead = 0;
	int32_t startIdx = 0;
	int32_t head = 0;
	int32_t size = length;
	uint32_t parsedPacketNum = 0;
	bool loopEndFlag = false;

	if (findHeader1 && _remain > 0)
	{
		memcpy(buf, this->_buf, this->_remain);
		head = this->_remain;
		this->_remain = 0;
		startIdx = 1;	// Header1は見つかっているのでその次の位置からHEADER2を探す
	}
	memcpy(&(buf[head]), readBytes, length);
	size += head;

	for (int32_t i = startIdx; (i < size) && !loopEndFlag; i++)
	{
		if (this->findHeader1 && !this->findHeader2)
		{
			if (buf[i] == MPCOM_HEADER2)
			{
				this->findHeader2 = true;
			}
			else
			{
				this->findHeader1 = false;
			}
		}
		else if (buf[i] == MPCOM_HEADER1)
		{
			packetHead = i;
			this->findHeader1 = true;
		}

		if (this->findHeader2)
		{
			if ((size - packetHead) < MPCOM_DMA_RX_DATA_SIZE)
			{
				loopEndFlag = true;
			}
			else
			{
				DMA_RX_DATA recvPacket;
				uint8_t packetBytes[MPCOM_DMA_RX_DATA_SIZE];

				memcpy(packetBytes, &(buf[packetHead]), MPCOM_DMA_RX_DATA_SIZE);
				if (this->parseData(packetBytes, &recvPacket))
				{
					memcpy(&rxData[parsedPacketNum], packetBytes, MPCOM_DMA_RX_DATA_SIZE);
					parsedPacketNum++;
				}
				else
				{
					// CRCエラー
//					cout << "CRC Error!" << endl;
				}

				i += MPCOM_DMA_RX_DATA_SIZE - 2;		// iはHeader2の位置であるので、次のパケットの先頭はi+MPCOM_DMA_RX_DATA_SIZE-1の位置。さらに次のループのi++があるため、-1しておく
				this->findHeader1 = false;
				this->findHeader2 = false;
			}
		}
	}

	if (this->findHeader1)
	{
		_remain = size - packetHead;
		memcpy(this->_buf, &buf[packetHead], this->_remain);
	}
	else
	{
		this->_remain = 0;
	}

	return parsedPacketNum;
}

/*
 * 受信データをパースし、構造体に格納する。CRCチェックもここで行う。
 */
bool CMasterControllerClient::parseData(uint8_t* packetBytes, DMA_RX_DATA* recvPacket)
{
	uint16_t crc;
	DMA_RX_DATA tempRecv;
	uint8_t bytes[MPCOM_DMA_RX_DATA_SIZE];

	memcpy(bytes, packetBytes, MPCOM_DMA_RX_DATA_SIZE);

	CCrc crcGen;
	crc = crcGen.calcCRC16(bytes, MPCOM_RX_DATA_SIZE);
	memcpy((void*)&tempRecv, bytes, MPCOM_DMA_RX_DATA_SIZE);
	if (crc == tempRecv.BUFF.DATA.CRC_CODE)
	{
		memcpy(packetBytes, &tempRecv, MPCOM_DMA_RX_DATA_SIZE);
//		printf("ok\n");
		return true;
	}
	else
	{
		return false;
	}
}


/*
 * 引数に渡された送信データオブジェクトをバイト配列に変換し、マスターコントローラに送る。
 */
void CMasterControllerClient::sendDataToMCU(CSendData* sendData) {
	uint8_t bytes[MPCOM_TX_DATA_SIZE];
	uint8_t* ret;
	// バイト配列の取得
	ret = sendData->getPacketBytes(bytes);
	// 送信データの表示（デバッグ用）
//	for(int i=0; i<MPCOM_DMA_TX_DATA_SIZE; i++){
//		printf("%02X, ", ret[i]);
//	}
	printf("\n");

	// 送信処理
	if(this->send(ret, MPCOM_DMA_TX_DATA_SIZE)==-1){
		printf("Serial Send Error!\n");
		exit(1);
	}
}

/*
 * 関節指令送信コマンド
 */
void CMasterControllerClient::setJointOrder(char* jointId, ePacketOrder order, CSendData* sendData) {
	int id;
	char *jids[MAX_JOINT_NUM];
	if (!strcmp(jointId, "all")){
		for (int i=0; i<MAX_JOINT_NUM; i++) sendData->setJointOrder(i, order);
	}
	else if (strstr(jointId, "/")){
		int cnt = this->strSplit(jointId, "/", jids);
		for (int i=0; i<cnt; i++){
			id = atoi(jids[i]) - 1;
			sendData->setJointOrder(id, order);
		}
	}
	else {
		id = atoi(jointId) - 1;
		sendData->setJointOrder(id, order);
	}
}

/*
 * 目標値指定コマンド
 */
void CMasterControllerClient::setValue(char* jointId, int valueNum, float value, CSendData* sendData) {
	int id;
	char *jids[MAX_JOINT_NUM];
	if (!strcmp(jointId, "all")){
		for(int i=0; i<MAX_JOINT_NUM; i++) {
			switch(valueNum) {
			case 1:
				sendData->setValue(i, 1, value);
				break;
			case 2:
				sendData->setValue(i, 2, value);
				break;
			case 3:
				sendData->setValue(i, 3, value);
				break;
			case 4:
				sendData->setValue(i, 4, value);
				break;
			}
		}
	} 
	else if (strstr(jointId, "/")){
		int cnt = this->strSplit(jointId, "/", jids);
		for (int i=0; i<cnt; i++){
			id = atoi(jids[i]) - 1;
			switch(valueNum) {
			case 1:
				sendData->setValue(id, 1, value);
				break;
			case 2:
				sendData->setValue(id, 2, value);
				break;
			case 3:
				sendData->setValue(id, 3, value);
				break;
			case 4:
				sendData->setValue(id, 4, value);
				break;
			}
		}
	}
	else {
		id = atoi(jointId) - 1;
		switch(valueNum) {
		case 1:
			sendData->setValue(id, 1, value);
			break;
		case 2:
			sendData->setValue(id, 2, value);
			break;
		case 3:
			sendData->setValue(id, 3, value);
			break;
		case 4:
			sendData->setValue(id, 4, value);
			break;
		}
	}
}

bool CMasterControllerClient::isEqual(const char* str1, const char* str2) {
	if(strcmp(str1, str2) == 0)
		return true;
	else
		return false;
}

int CMasterControllerClient::strSplit(char* str, const char* delim, char* res[]){
	char *tk;
	int cnt=0;
	tk = strtok(str, delim);
	while(tk!=NULL && cnt<MAX_JOINT_NUM) {
		res[cnt] = tk;
		cnt++;
		tk = strtok(NULL, delim);
	}
	return cnt;
}


/* ======================================
 * Public
 * ====================================== */
/*
 * 最新状態の受信
 */
string CMasterControllerClient::execCommand(char* commandJson) {
	char resJsonStr[1000];

	Json::Reader reader;
	Json::Value root;
	bool b = reader.parse(commandJson, root);
	const char* command = root["command"].asCString();

	if(isEqual("--state", command)) {
		this->showCuurentStatus();
	} else if (isEqual("--servo", command)) {	// Change Servo Status ===========
		char* servoState	= (char*)root["servo_state"].asCString();
		char* jointId		= (char*)root["joint_id"].asCString();
		if(isEqual("on", servoState) || isEqual("ON", servoState)) {
			this->sendServoOn(jointId);
		} else if (isEqual("off", servoState) || isEqual("OFF", servoState)) {
			this->sendServoOff(jointId);
		}
	} else if (isEqual("--brake", command)) {	// Change Brake Status ===========
		char* brakeState	= (char*)root["brake_state"].asCString();
		char* jointId		= (char*)root["joint_id"].asCString();
		if(isEqual("on", brakeState) || isEqual("ON", brakeState)) {
			this->sendBrakeOn(jointId);
		} else if (isEqual("off", brakeState) || isEqual("OFF", brakeState)) {
			this->sendBrakeOff(jointId);
		}
	} else if (isEqual("--quit", command)) {	// Set Servo Off =================
		this->sendServoOff("all");
	} else if (isEqual("--reset", command)) {	// Reset Joint ===================
		char* jointId		= (char*)root["joint_id"].asCString();
		this->sendReset(jointId);
	} else if (isEqual("--mode", command)) {	// Change Control Mode ===========
		char* modeId	= (char*)root["mode_id"].asCString();
		char* jointId	= (char*)root["joint_id"].asCString();
		this->sendCtrlMode(jointId, (unsigned char)atoi(modeId));
	} else if (isEqual("--kp", command)) {		// Change Kp Parameter ===========
		char* jointId	= (char*)root["joint_id"].asCString();
		char* value		= (char*)root["value"].asCString();
		this->sendParamKp(jointId, atof(value));
	} else if (isEqual("--ki", command)) {		// Change Ki Parameter ===========
		char* jointId	= (char*)root["joint_id"].asCString();
		char* value		= (char*)root["value"].asCString();
		this->sendParamKi(jointId, atof(value));
	} else if (isEqual("--kd", command)) {		// Change Kd Parameter ===========
		char* jointId	= (char*)root["joint_id"].asCString();
		char* value		= (char*)root["value"].asCString();
		this->sendParamKd(jointId, atof(value));
	} else if (isEqual("--da", command)) {		// Change Damper Parameter =======
		char* jointId	= (char*)root["joint_id"].asCString();
		char* value		= (char*)root["value"].asCString();
		this->sendParamDa(jointId, atof(value));
	} else if (isEqual("--cur", command)) {		// Change Reference Current ======
		char* jointId	= (char*)root["joint_id"].asCString();
		char* value		= (char*)root["value"].asCString();
		this->sendRefCurrent(jointId, atof(value));
	} else if (isEqual("--pos", command)) {		// Change Reference Position =====
		char* jointId	= (char*)root["joint_id"].asCString();
		char* value		= (char*)root["value"].asCString();
		this->sendRefPosition(jointId, atof(value));
	} else if (isEqual("--vel", command)) {		// Change Reference Velocity =====
		char* jointId	= (char*)root["joint_id"].asCString();
		char* value		= (char*)root["value"].asCString();
		this->sendRefVelocity(jointId, atof(value));
	} else if (isEqual("--tor", command)) {		// Change Reference Torque =======
		char* jointId	= (char*)root["joint_id"].asCString();
		char* value		= (char*)root["value"].asCString();
		this->sendRefTorque(jointId, atof(value));
	} else if (isEqual("--tptl", command)) {
		char* jointId	= (char*)root["joint_id"].asCString();
		float pos		= atof((char*)root["pos"].asCString());
		float transitionTime = atof(root["time"].asCString());
		this->sendTrajectoryPTLinear(jointId, pos, transitionTime);
	} else if (isEqual("--tpts", command)) {
		char* jointId	= (char*)root["joint_id"].asCString();
		float pos		= atof((char*)root["pos"].asCString());
		float transitionTime = atof(root["time"].asCString());
		this->sendTrajectoryPTSin(jointId, pos, transitionTime);
	} else if (isEqual("--tpvt", command)) {
		char* filename = (char*)root["csv_file"].asCString();
		this->sendTrajectoryPVT(filename);
	} else if (isEqual("--ts", command)) {
		char* jointId	= (char*)root["joint_id"].asCString();
		this->startTrajectoryControl(jointId);
	} else if (isEqual("--tc", command)) {
		char* jointId	= (char*)root["joint_id"].asCString();
		this->clearMasterTrajectoryViaPointsBuffer(jointId);
	}

/*
	else if (isEqual("--tr", command)) {		// Set Trajectory ================
		char* jointId	= (char*)root["joint_id"].asCString();
		char* pos		= (char*)root["pos"].asCString();
		if(root["time"].type() == Json::stringValue) {
			this->setTrajectory(jointId, atof(pos), atof(root["time"].asCString()));
		}else{
			this->setAngTrajectory(jointId, atof(pos));
		}
	} else if (isEqual("--exec-tr", command)) {	// Execute Trajectory Control ====
		char* jointId	= (char*)root["joint_id"].asCString();
		this->startTrafectoryControl(jointId);
	} else if (isEqual("--pvt", command)) {		// Set PVT Points csv file =======
		char* filename = (char*)root["csv_file"].asCString();
		this->sendPvtPoints(filename);
	} else if (isEqual("--pvt-direct", command)) { // Set PVT Points JSON file

	} else if (isEqual("--exec-pvt", command)) {// Execute PVT Control ===========
		this->startPvtControl();
	} else if (isEqual("--clear-pvt", command)){// Clear PVT Buffer
		this->clearPvtBuffer();
	}*/

	return getCurStateJson();
}

string CMasterControllerClient::getCurStateJson() {
	Json::Value curState;
	Json::Value jointState[MAX_JOINT_NUM];
	Json::Value vec(Json::arrayValue);

	curState["timestamp"] = std::to_string(this->curState.BUFF.DATA.timeStamp);
	curState["hostTimestamp"] = std::to_string(this->curState.BUFF.DATA.hostTimeStamp);
	curState["duration"] = this->curState.BUFF.DATA.duration;

	for(int i =0; i<MAX_JOINT_NUM; i++) {
		jointState[i]["joint_id"] = i + 1;
		jointState[i]["type"] = this->curState.BUFF.DATA.joint[i].type;
		jointState[i]["comStatus"] = this->curState.BUFF.DATA.joint[i].comStatus;
		jointState[i]["crtlMode"] = this->curState.BUFF.DATA.joint[i].ctrlMode;
		jointState[i]["currect"] =this->curState.BUFF.DATA.joint[i].current;
		jointState[i]["damper"] = this->curState.BUFF.DATA.joint[i].damper;
//		jointState[i]["ewStatus"] = this->curState.BUFF.DATA.joint[i].ewStatus;
		jointState[i]["kd"] = this->curState.BUFF.DATA.joint[i].kd;
		jointState[i]["ki"] = this->curState.BUFF.DATA.joint[i].ki;
		jointState[i]["kp"] = this->curState.BUFF.DATA.joint[i].kp;
		jointState[i]["position"] = this->curState.BUFF.DATA.joint[i].position;
		jointState[i]["trjStatus"] = this->curState.BUFF.DATA.joint[i].trjStatus;
		jointState[i]["trjViaRemain"] = this->curState.BUFF.DATA.joint[i].trjViaRemain;
		// jointState[i]["pvtRemain"] = this->curState.BUFF.DATA.joint[i].pvtRemain;
		// jointState[i]["pvtStatus"] = this->curState.BUFF.DATA.joint[i].pvtStatus;
		jointState[i]["refCurrent"] = this->curState.BUFF.DATA.joint[i].refCurrent;
		jointState[i]["refPosition"] = this->curState.BUFF.DATA.joint[i].refPosition;
		jointState[i]["refTorque"] = this->curState.BUFF.DATA.joint[i].refTorque;
		jointState[i]["refVelocity"] = this->curState.BUFF.DATA.joint[i].refVelocity;
		jointState[i]["systemMode"] = this->curState.BUFF.DATA.joint[i].systemMode;
		jointState[i]["temperature"] = this->curState.BUFF.DATA.joint[i].temperature;
		jointState[i]["torque"] = this->curState.BUFF.DATA.joint[i].torque;
		jointState[i]["trjStatus"] = this->curState.BUFF.DATA.joint[i].trjStatus;
		jointState[i]["velocity"] = this->curState.BUFF.DATA.joint[i].velocity;
		vec.append(jointState[i]);
	}

	curState["jointState"] = vec;

	return curState.toStyledString();
}

/*
 * 最新状態の受信
 */
uint16_t CMasterControllerClient::receiveStatus() {
	uint16_t parsedNum = 0;
	DMA_RX_DATA rxData[100];
	unsigned char buf[BUF_SIZE];

	// Receive Data from Serial Port
	int len = this->recv(buf, sizeof(buf));
	if(len==-1) {
		printf("Return -1\n");
		printf("[Error] Cannot Receive data from Serial Port. Please check the connection.\n");
		exit(1);
	} else if (len!=0){
		parsedNum = this->searchPacket((uint8_t*)buf, len, &rxData[0]);
		if(parsedNum!=0){
			this->curState = rxData[parsedNum-1];
		}
	}

	// Return Data Number
	return parsedNum;
}

/*
 * 状態表示コマンド
 */
void CMasterControllerClient::showCuurentStatus() {
	uint16_t parsedNum = 0;
	uint32_t allRecvNum = 0;

	printf("-------------------------------\n");
	printf("timestamp: %llu\n", this->curState.BUFF.DATA.timeStamp);
	printf("-------------------------------\n");
	printf("JOINT\tSTATUS\tSYS\tCTRL\tTRJ\tPVT\tPVT_REMAIN\n");
	for(int j=0; j<MAX_JOINT_NUM; j++){
		printf("%d\t%02X\t%02X\t%02X\t%02X\t%02X\t",
				j,
				this->curState.BUFF.DATA.joint[j].comStatus,
				this->curState.BUFF.DATA.joint[j].systemMode,
				this->curState.BUFF.DATA.joint[j].ctrlMode,
				this->curState.BUFF.DATA.joint[j].trjStatus);
		printf("%" PRIu16 "\n", this->curState.BUFF.DATA.joint[j].trjViaRemain);
	}
	printf("-------------------------------\n");
	printf("JOINT\tKP\tKI\tKD\tDMP\n");
	for(int j=0; j<MAX_JOINT_NUM; j++){
		printf("%d\t%.02f\t%.02f\t%.02f\t%.02f\n",
				j,
				this->curState.BUFF.DATA.joint[j].kp,
				this->curState.BUFF.DATA.joint[j].ki,
				this->curState.BUFF.DATA.joint[j].kd,
				this->curState.BUFF.DATA.joint[j].damper);
	}
	printf("-------------------------------\n");
	printf("JOINT\tPOS\tVEL\tCUR\tTRQ\tTEMP\n");
	for(int j=0; j<MAX_JOINT_NUM; j++){
		printf("%d\t%.02f\t%.02f\t%.02f\t%.02f\t%.02f\n",
				j,
				this->curState.BUFF.DATA.joint[j].position,
				this->curState.BUFF.DATA.joint[j].velocity,
				this->curState.BUFF.DATA.joint[j].current,
				this->curState.BUFF.DATA.joint[j].torque,
				this->curState.BUFF.DATA.joint[j].temperature);
	}
}


/*
 * サーボONコマンド
 */
void CMasterControllerClient::sendServoOn(char* jointId) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_SERVO_ON, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * サーボOFFコマンド
 */
void CMasterControllerClient::sendServoOff(char* jointId) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_SERVO_OFF, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * Brake ONコマンド
 */
void CMasterControllerClient::sendBrakeOn(char* jointId) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_BRAKE_ON, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * Brake OFFコマンド
 */
void CMasterControllerClient::sendBrakeOff(char* jointId) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_BRAKE_OFF, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * リセットコマンド
 */
void CMasterControllerClient::sendReset(char* jointId) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_RESET, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * 制御モード指定コマンド
 */
void CMasterControllerClient::sendCtrlMode(char* jointId, unsigned char ctrlMode) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_RUN_MODE, sendData);
	this->setValue(jointId, 1, (float)ctrlMode, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * Kpゲイン設定コマンド
 */
void CMasterControllerClient::sendParamKp(char* jointName, float kp) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointName, PACKET_ORDER_PARAM_KP, sendData);
	this->setValue(jointName, 1, kp, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * Kiゲイン設定コマンド
 */
void CMasterControllerClient::sendParamKi(char* jointName, float ki) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointName, PACKET_ORDER_PARAM_KI, sendData);
	this->setValue(jointName, 1, ki, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * Kdゲイン設定コマンド
 */
void CMasterControllerClient::sendParamKd(char* jointName, float kd) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointName, PACKET_ORDER_PARAM_KD, sendData);
	this->setValue(jointName, 1, kd, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * Dumper設定コマンド
 */
void CMasterControllerClient::sendParamDa(char* jointName, float da) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointName, PACKET_ORDER_PARAM_DAMPER, sendData);
	this->setValue(jointName, 1, da, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * 目標電流値設定コマンド
 */
void CMasterControllerClient::sendRefCurrent(char* jointId, float refCur) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_CURRENT, sendData);
	this->setValue(jointId, 1, refCur, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * 目標姿勢設定コマンド
 */
void CMasterControllerClient::sendRefPosition(char* jointId, float refPos) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_POSITION, sendData);
	this->setValue(jointId, 1, refPos, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * 目標速度設定コマンド
 */
void CMasterControllerClient::sendRefVelocity(char* jointId, float refVel){
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_VELOCITY, sendData);
	this->setValue(jointId, 1, refVel, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

/*
 * 目標トルク設定コマンド
 */
void CMasterControllerClient::sendRefTorque(char* jointId, float refTrq){
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_TORQUE, sendData);
	this->setValue(jointId, 1, refTrq, sendData);
	// Send
//	sendData->showSendData();
	this->sendDataToMCU(sendData);
}

// New Trajectory Control
/*
 * 台形補間
 */
void CMasterControllerClient::sendTrajectoryPTLinear(char* jointId, float refPos, float transitionTime) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_TRAJ_VIA_APPEND_PT_LINEAR, sendData);
	this->setValue(jointId, 1, refPos, sendData);
	this->setValue(jointId, 2, transitionTime, sendData);
	this->sendDataToMCU(sendData);
}

/*
 * SIN補間
 */
void CMasterControllerClient::sendTrajectoryPTSin(char* jointId, float refPos, float transitionTime) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_TRAJ_VIA_APPEND_PT_SIN, sendData);
	this->setValue(jointId, 1, refPos, sendData);
	this->setValue(jointId, 2, transitionTime, sendData);
	this->sendDataToMCU(sendData);
}

/*
 * PVT点送信コマンド（単関節）
 */
void CMasterControllerClient::setTrajectoryPVT(char* jointId, float position, float velocity, float time) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_TRAJ_VIA_APPEND_PVT, sendData);
	this->setValue(jointId, 1, position, sendData);
	this->setValue(jointId, 2, velocity, sendData);
	this->setValue(jointId, 3, time, sendData);
	this->sendDataToMCU(sendData);
}

/*
 * PVT点送信コマンド（複数関節同時設定）
 */
void CMasterControllerClient::setTrajectoryPVT(list<PVT_POINT> pvtPointList) {
	if(!pvtPointList.empty()){
		CSendData* sendData = new CSendData();
		list<PVT_POINT>::iterator pvtItr = pvtPointList.begin();
		while(pvtItr != pvtPointList.end()){
			PVT_POINT pvtPoint = *pvtItr;
			this->setJointOrder(pvtPoint.jointId, PACKET_ORDER_TRAJ_VIA_APPEND_PVT, sendData);
			this->setValue(pvtPoint.jointId, 1, pvtPoint.position, sendData);
			this->setValue(pvtPoint.jointId, 2, pvtPoint.velocity, sendData);
			this->setValue(pvtPoint.jointId, 3, pvtPoint.time, sendData);
			printf("[AfterSet-List] %s, %f, %f\n", pvtPoint.jointId, pvtPoint.position, pvtPoint.velocity);
			pvtItr++;
		}
		this->sendDataToMCU(sendData);
	}else{
		printf("Empty!\n");
	}
}

/*
 * PVT設定コマンド（複数関節同時設定）
 */
void CMasterControllerClient::setTrajectoryPVT(PVT_POINT* pvtPoints) {
	CSendData* sendData = new CSendData();
	for(int i=0; i<MAX_JOINT_NUM; i++){
		this->setJointOrder(pvtPoints[i].jointId, PACKET_ORDER_TRAJ_VIA_APPEND_PVT, sendData);
		this->setValue(pvtPoints[i].jointId, 1, pvtPoints[i].position, sendData);
		this->setValue(pvtPoints[i].jointId, 2, pvtPoints[i].velocity, sendData);
		this->setValue(pvtPoints[i].jointId, 3, pvtPoints[i].time, sendData);
	}
	printf("PvtStep: %f\n", pvtPoints[0].time, sendData);
	this->sendDataToMCU(sendData);
}

/*
 * PVT設定送信コマンド
 */
void CMasterControllerClient::sendTrajectoryPVT(char* filepath) {
	this->pvtPointList = CCSVLoader::getDataList(filepath, ',');
	this->setPvtPointSending(true);
	printf("[New PVT Point] %ld\n", this->pvtPointList.size());
}

/*
 * マスタコントローラのバッファ開放コマンド
 */
void CMasterControllerClient::clearMasterTrajectoryViaPointsBuffer(char* jointId) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_TRAJ_VIA_CLEAR, sendData);
	// バッファ開放時にPC側の送信待ちバッファも開放する。
	if(!this->pvtPointList.empty()){
		this->pvtPointList.clear();
	}
	this->sendDataToMCU(sendData);
}

/*
 * TRAJ制御開始コマンド
 */
void CMasterControllerClient::startTrajectoryControl(char* jointId) {
	CSendData* sendData = new CSendData();
	this->setJointOrder(jointId, PACKET_ORDER_TRAJ_CTRL_START, sendData);
	this->sendDataToMCU(sendData);
}

// /*
//  * 軌道設定コマンド
//  */
// void CMasterControllerClient::setTrajectory(char* jointId, float refPos, float transitionTime){
// 	printf("%s: %.02f[deg], %.02f[sec]\n", jointId, refPos, transitionTime);
//
// 	CSendData* sendData = new CSendData();
// 	this->setJointOrder(jointId, PACKET_ORDER_TRAJ, sendData);
// 	this->setValue(jointId, 1, refPos, sendData);
// 	this->setValue(jointId, 2, transitionTime, sendData);
// 	// Send
// //	sendData->showSendData();
// 	this->sendDataToMCU(sendData);
// }
//
// /*
//  * 軌道設定コマンド（関節角度のみ）
//  */
// void CMasterControllerClient::setAngTrajectory(char* jointId, float refPos){
// 	printf("%s: %.02f[deg]\n", jointId, refPos);
//
// 	CSendData* sendData = new CSendData();
// 	this->setJointOrder(jointId, PACKET_ORDER_ANG_TRAJ, sendData);
// 	this->setValue(jointId, 1, refPos, sendData);
// 	// Send
// //	sendData->showSendData();
// 	this->sendDataToMCU(sendData);
// }
//
// /*
//  * 軌道制御開始コマンド
//  */
// void CMasterControllerClient::startTrafectoryControl(char* jointId){
// 	CSendData* sendData = new CSendData();
// 	this->setJointOrder(jointId, PACKET_ORDER_TRAJ_CTRL_START, sendData);
// 	// Send
// //	sendData->showSendData();
// 	this->sendDataToMCU(sendData);
// }
//
// /*
//  * PVT点送信コマンド（単関節）
//  */
// void CMasterControllerClient::setPvtPoints(char* jointId, float position, float velocity, float time){
// 	CSendData* sendData = new CSendData();
// 	this->setJointOrder(jointId, PACKET_ORDER_PVT_POINT_APPEND, sendData);
// 	this->setValue(jointId, 1, position, sendData);
// 	this->setValue(jointId, 2, velocity, sendData);
// 	this->setValue(jointId, 3, time, sendData);
// 	this->sendDataToMCU(sendData);
// }
//
// /*
//  * PVT設定コマンド（複数関節同時設定）
//  */
// void CMasterControllerClient::setPvtPoints(list<PVT_POINT> pvtPointList){
// 	if(!pvtPointList.empty()){
// 		CSendData* sendData = new CSendData();
// 		list<PVT_POINT>::iterator pvtItr = pvtPointList.begin();
// 		while(pvtItr != pvtPointList.end()){
// 			PVT_POINT pvtPoint = *pvtItr;
// 			this->setJointOrder(pvtPoint.jointId, PACKET_ORDER_PVT_POINT_APPEND, sendData);
// 			this->setValue(pvtPoint.jointId, 1, pvtPoint.position, sendData);
// 			this->setValue(pvtPoint.jointId, 2, pvtPoint.velocity, sendData);
// 			this->setValue(pvtPoint.jointId, 3, pvtPoint.time, sendData);
// 			printf("[AfterSet-List] %s, %f, %f\n", pvtPoint.jointId, pvtPoint.position, pvtPoint.velocity);
// 			pvtItr++;
// 		}
// //		sendData->showSendData();
// 		this->sendDataToMCU(sendData);
// 	}else{
// 		printf("Empty!\n");
// 	}
// }
//
// /*
//  * PVT設定コマンド（複数関節同時設定）
//  */
// void CMasterControllerClient::setPvtPoints(PVT_POINT* pvtPoints){
// 	CSendData* sendData = new CSendData();
// 	for(int i=0; i<MAX_JOINT_NUM; i++){
// 		this->setJointOrder(pvtPoints[i].jointId, PACKET_ORDER_PVT_POINT_APPEND, sendData);
// 		this->setValue(pvtPoints[i].jointId, 1, pvtPoints[i].position, sendData);
// 		this->setValue(pvtPoints[i].jointId, 2, pvtPoints[i].velocity, sendData);
// 		this->setValue(pvtPoints[i].jointId, 3, pvtPoints[i].time, sendData);
// 	}
// //	sendData->showSendData();
// 	printf("PvtStep: %f\n", pvtPoints[0].time, sendData);
// 	this->sendDataToMCU(sendData);
// }
//
// /*
//  * PVT設定送信コマンド
//  */
// void CMasterControllerClient::sendPvtPoints(char* filepath){
// 	this->pvtPointList = CCSVLoader::getDataList(filepath, ',');
// 	this->setPvtPointSending(true);
// 	printf("[New PVT Point] %ld\n", this->pvtPointList.size());
// }
//
// /*
//  * PVT制御開始コマンド
//  */
// void CMasterControllerClient::startPvtControl(){
// 	CSendData* sendData = new CSendData();
// 	this->setJointOrder("all", PACKET_ORDER_PVT_START, sendData);
// //	sendData->showSendData();
// 	this->sendDataToMCU(sendData);
// }
//
// /*
//  * マスタコントローラのPVTバッファ開放コマンド
//  */
// void CMasterControllerClient::clearPvtBuffer() {
// 	CSendData* sendData = new CSendData();
// 	this->setJointOrder("all", PACKET_ORDER_PVT_CLEAR, sendData);
// 	// バッファ開放時にPC側の送信待ちバッファも開放する。
// 	if(!this->pvtPointList.empty()){
// 		this->pvtPointList.clear();
// 	}
// //	sendData->showSendData();
// 	this->sendDataToMCU(sendData);
// }
