/*
 * CSerialPort.cpp
 *
 *  Created on: 2016/09/17
 *      Author: vivitter
 */

#include "CSerialPort.h"

#ifdef __linux__
#include <linux/serial.h>
#endif
#include <stddef.h>
#include <err.h>
using namespace std;

/*
 * Constructor 2
 * 接続処理も含んだコンストラクタ
 */
CSerialPort::CSerialPort(char* deviceName, unsigned long baudRate) {
	this->fd = 0;
	this->deviceName = NULL;
	this->baudRate = 0;
	this->connect(deviceName, baudRate);
}

/*
 * Constructor
 */
CSerialPort::CSerialPort() {
	this->fd = 0;
	this->deviceName = NULL;
	this->baudRate = 0;
}

/*
 * Destructor
 */
CSerialPort::~CSerialPort() {
}

/*
 * 接続処理
 */
#ifdef __linux__
int CSerialPort::rate_to_constant(unsigned long baudrate) {
#define B(x) case x: return B##x
	switch(baudrate) {
		B(50);     B(75);     B(110);    B(134);    B(150);
		B(200);    B(300);    B(600);    B(1200);   B(1800);
		B(2400);   B(4800);   B(9600);   B(19200);  B(38400);
		B(57600);  B(115200); B(230400);
	default: return 0;
	}
#undef B
}
int CSerialPort::connect(char* device, unsigned long rate)
{
	struct termios options;
	struct serial_struct serinfo;
	int speed = 0;

	/* Open and configure serial port */
	if ((fd = open(device,O_RDWR|O_NOCTTY)) == -1) {
		printf("Serial Port Open Error!\n");
		return -1;
	}

	speed = this->rate_to_constant(rate);

	if (speed == 0) {
		/* Custom divisor */
		serinfo.reserved_char[0] = 0;
		if (ioctl(fd, TIOCGSERIAL, &serinfo) < 0)
			return -1;
		serinfo.flags &= ~ASYNC_SPD_MASK;
		serinfo.flags |= ASYNC_SPD_CUST;
		serinfo.custom_divisor = (serinfo.baud_base + (rate / 2)) / rate;
		if (serinfo.custom_divisor < 1)
			serinfo.custom_divisor = 1;
		if (ioctl(fd, TIOCSSERIAL, &serinfo) < 0)
			return -1;
		if (ioctl(fd, TIOCGSERIAL, &serinfo) < 0)
			return -1;
		if (serinfo.custom_divisor * rate != serinfo.baud_base) {
			warnx("actual baudrate is %d / %d = %f",
			      serinfo.baud_base, serinfo.custom_divisor,
			      (float)serinfo.baud_base / serinfo.custom_divisor);
		}
	}

	fcntl(fd, F_SETFL, 0);
	tcgetattr(fd, &options);
	cfsetispeed(&options, speed ?: B38400);
	cfsetospeed(&options, speed ?: B38400);
	cfmakeraw(&options);
	options.c_cflag |= (CLOCAL | CREAD | CS8);
	options.c_cflag &= ~PARENB;
	options.c_cflag &= ~CSTOPB;
	options.c_cflag &= ~CSIZE;
	options.c_cflag &= ~CRTSCTS;
	options.c_iflag &= ~IXON;
	if (tcsetattr(fd, TCSANOW, &options) != 0)
		return -1;

	return fd;
}
#elif __APPLE__
int CSerialPort::connect(char* deviceName, unsigned long baudRate) {
	this->deviceName = deviceName;
	this->baudRate = baudRate;

	// Open Serial Port
	this->fd = open(deviceName, O_RDWR | O_NOCTTY);
	// Error handling
	if(this->fd < 0) {
		cout << deviceName << " does not exist." << endl;
		return -1;
	}

	// Get Current Settings
	tcgetattr(fd, &this->oldtio);
	this->newtio = this->oldtio;

	// Set baud rate & Other options
	cfsetspeed(&this->newtio, baudRate);
	newtio.c_cflag = CS8;
	newtio.c_iflag = IXOFF;

	// Apply new settings
	tcflush(fd, TCIFLUSH);
	tcsetattr(fd, TCSANOW, &this->newtio);

	return fd;
}
#endif


/*
 * 切断処理
 */
void CSerialPort::disconnect(void) {
	// Return to initial settings
	tcsetattr(this->fd, TCSANOW, &this->oldtio);
	// Close Serial Port
	close(this->fd);
}

/*
 * 送信処理
 */
int CSerialPort::send(unsigned char* buffer, int bufSize) {
	int len = write(this->fd, buffer, bufSize);
	if(len!=bufSize)
		return -1;

	return len;
}


/*
 * 受信処理
 */
int CSerialPort::recv(unsigned char* buffer, int bufSize) {
	int len = read(this->fd, buffer, bufSize);
	if(len == 0)
//		cout << "Blank Data" << endl;
	if(len < 0) {
		cout << "Serial Read Error!" << endl;
		exit(2);
	}

	return len;
}
