//============================================================================
// Name        : ToroboArmManager.cpp
// Author      : Kohei Kojima
// Version     :
// Copyright   : Tokyo Robotics
// Description : Hello World in C, Ansi-style
//============================================================================

#include <zmq.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <pthread.h>
#include "zhelpers.h"
#include <time.h>
#include <sys/time.h>

#ifdef __linux__
#include <jsoncpp/json/json.h>
#elif __APPLE__
#include <json/json.h>
#endif

#include "CMasterControllerClient.h"

#define SERIAL_PORT "/dev/ttyUSB0"
#define BAUD_RATE 3000000

/*
 * 状態更新スレッド
 * アームの最新状態をシリアルポートから取得する。PVT点の送信処理も行う。
 */
void *pub_thread(void *ptr)
{
	CMasterControllerClient* client = (CMasterControllerClient *)ptr;
	printf("Publisher Launched!\n");

	//  Prepare our context and publisher
	void *context = zmq_ctx_new ();
	void *publisher = zmq_socket (context, ZMQ_PUB);
	zmq_bind (publisher, "tcp://*:5554");

	int sendCounter = 0;

	while (1) {
		// 現在値の取得
		// 周期ごとにクライアント内の現在値を更新する。
		client->receiveStatus();

		// JSON文字列形式で現在値を取得する。
		string curStateJson = client->getCurStateJson();

		// 現在値をZeroMQのソケットにパブリッシュする。
		s_sendmore(publisher, "ToroboArmManager");
		s_send(publisher, (char*)client->getCurStateJson().c_str());

		// PVT送信ステータスが「送信中」の場合
		if(client->isPvtPointSending()) {
			// CSVファイルから読み込んだPVTリストが空ではない場合
			if(!client->pvtPointList.empty()){
				// １行分の状態を記録する配列
				PVT_POINT pvtPoints[MAX_JOINT_NUM];

				// 対象の１行のtimeを記録するバッファ
				float timeBuff;

				// データ全体から先頭行のイテレータを取得する。
				list<list<float>>::iterator dataItr = client->pvtPointList.begin();
				// 先頭行の冒頭のカラムのイテレータを取得する。
				list<float> lineList = *dataItr;
				list<float>::iterator lineItr = lineList.begin();

				// 関節ごとにバッファの状態を確認しつつ、全関節コントローラが受信可能な場合のみ送信する。
				// バッファがいっぱいの場合は次の周期まで待つ。
				for(int i=0; i<MAX_JOINT_NUM; i++) {
					if(client->getCurState().BUFF.DATA.joint[i].trjViaRemain < PVT_BUFF_SIZE-1) {
						// Joint ID
						sprintf(pvtPoints[i].jointId, "%d", i+1);

						// Time（行の冒頭にしか書かれていない）
						if(i==0) {
							pvtPoints[i].time = *lineItr;
							timeBuff = *lineItr;
							lineItr++;
						} else {
							pvtPoints[i].time = timeBuff;
						}
						// Position
						pvtPoints[i].position = *lineItr;
						lineItr++;
						// Velocity
						pvtPoints[i].velocity = *lineItr;
						lineItr++;

						// 全ての関節のバッファに余裕がある場合
						if(i == MAX_JOINT_NUM-1){
							// Master Controllerに行のデータを送信
							// client->setPvtPoints(pvtPoints);
							client->setTrajectoryPVT(pvtPoints);
							// 送信した先頭行のデータを削除する。
							client->pvtPointList.pop_front();
							sendCounter++;
						}
					}else{
						// バッファが溢れている関節がある場合、送信せずにスキップする。
						printf("Master Controller's Buffer is full.\n");
						printf("Waiting...\n");
						break;
					}
				}
			}else{
				// PVT送信ステータスが「送信中」でデータリストが空の場合、送信完了とする。
				printf("All PVT Points Send\n");
				client->setPvtPointSending(false);
				sendCounter = 0;
			}
		}

		// 0.05秒待つ
		usleep(10000);
	}

	//  We never get here, but clean up anyhow
	zmq_close (publisher);
	zmq_ctx_destroy (context);
}

/*
 * コマンド待受スレッド
 * ToroboArmCommandInterfaceからのコマンドを受け取り、必要な処理を実行する。
 */
void *recv_command_thread(void *ptr)
{
	CMasterControllerClient* client = (CMasterControllerClient *)ptr;
	printf("Command Receiver Launched!\n");

	//  Socket to talk to clients
	void *context = zmq_ctx_new ();
	void *responder = zmq_socket (context, ZMQ_REP);
	int rc = zmq_bind (responder, "tcp://*:5555");
	assert (rc == 0);

	// Command Execution Loop
	while (1) {
		char commandJson[256]="";

		// Wait for command sending
		zmq_recv (responder, commandJson, 256, 0);
		printf("Command: %s\n", commandJson);

		// Execute Command
		string curStateJson = client->execCommand(commandJson);

		// Send Result to Client
		zmq_send (responder, curStateJson.c_str(), curStateJson.length(), 0);
	}
}

int main(int argc, char* argv[]) {
	pthread_t thread1, thread2;
	int iret1, iret2;

	CMasterControllerClient* client = new CMasterControllerClient(SERIAL_PORT, BAUD_RATE);

	// Create Threads
	iret1 = pthread_create(&thread1, NULL, pub_thread, client);
	iret2 = pthread_create(&thread2, NULL, recv_command_thread, client);

	pthread_join(thread1, NULL);
	pthread_join(thread2, NULL);

	printf("Thread 1 returns: %d\n", iret1);
	printf("Thread 2 returns: %d\n", iret2);
	return 0;
}
